import 'dart:async';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:bluetooth_print_plus/bluetooth_print_plus.dart';
import 'package:ovosale/app/package/bluetooth_print/bluetooth_print.dart';
import 'package:ovosale/core/helper/string_format_helper.dart';
import 'package:ovosale/core/utils/url_container.dart';
import 'package:ovosale/core/utils/util.dart';
import 'package:ovosale/data/controller/billing_section/billing_section_controller.dart';
import 'package:ovosale/data/controller/cart/global_controller.dart';
import 'package:ovosale/data/controller/checkout/checkout_controller.dart';
import 'package:ovosale/data/controller/file_download/file_downloader.dart';
import 'package:ovosale/data/model/customer/customer_response_model.dart';
import 'package:ovosale/data/model/new_sale/confirm_sale_response_model.dart';
import 'package:ovosale/data/model/payment_method/payment_account_resonse_model.dart';
import 'package:ovosale/data/model/payment_method/payment_method_response_model.dart';
import 'package:ovosale/data/model/payment_method/view_sale_response_model.dart'
    as prefix;
import 'package:ovosale/data/repo/payment/payment_repo.dart';
import 'package:ovosale/data/services/shared_pref_service.dart';

import '../../../app/components/snack_bar/show_custom_snackbar.dart';
import '../../../core/helper/date_converter.dart';
import '../../../core/utils/my_strings.dart';
import '../../model/global/response_model/response_model.dart';
import '../../model/new_sale/new_sale_response_model.dart';
import '../../model/profile/profile_response_model.dart';

// Import your BluetoothPrintService (adjust path as needed)

class PaymentController extends GetxController {
  PaymentRepo paymentRepo;
  ProfileResponseModel model = ProfileResponseModel();

  PaymentController({required this.paymentRepo});

  TextEditingController noteController = TextEditingController();

  BillingSectionController billingController = Get.find();

  bool isLoading = false;
  String saleId = "";
  prefix.SaleDataModel? saleDataModel;
  prefix.CompanyInformation? companyInformation;
  String defaultCurrency = "";

  List<PaymentMethod> paymentMethodList = [];

  PaymentMethod selectedPaymentMethod = PaymentMethod();
  CustomerData customerData = CustomerData();
  PaymentAccountModel selectedPaymentAccount = PaymentAccountModel();

  // Updated Bluetooth service integration
  final BluetoothPrintService bluetoothPrintService =
      BluetoothPrintService.instance;
  BluetoothDevice? selectedDevice;
  bool connected = false;
  String statusMessage = 'No device connected';
  List<PaymentAccountModel> paymentAccountsDataList = [];
  // Stream subscriptions for proper cleanup
  StreamSubscription? _stateSubscription;
  StreamSubscription? _scanningSubscription;
  StreamSubscription? _scanResultsSubscription;

  @override
  void onInit() {
    super.onInit();
    initBluetooth();
  }

  @override
  void onClose() {
    // Clean up all subscriptions
    _stateSubscription?.cancel();
    _scanningSubscription?.cancel();
    _scanResultsSubscription?.cancel();
    super.onClose();
  }

  getPaymentMethod() async {
    isLoading = true;
    update();

    ResponseModel responseModel = await paymentRepo.getPaymentMethod();

    if (responseModel.statusCode == 200) {
      PaymentMethodResponseModel model =
          PaymentMethodResponseModel.fromJson(responseModel.responseJson);

      if (model.status.toString() ==
          MyStrings.success.toString().toLowerCase()) {
        paymentMethodList = model.data?.paymentMethods ?? [];

        defaultCurrency = SharedPreferenceService.getString(
          SharedPreferenceService.defaultCurrencyKey,
        );
      } else {
        CustomSnackBar.error(
            errorList: model.message ?? [MyStrings.requestFail.tr]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message.tr]);
    }

    isLoading = false;
    update();
  }

  viewSaleData({required ConfirmSaleResponseModel model}) async {
    if (model.status.toString() == MyStrings.success.toString().toLowerCase()) {
      saleDataModel = model.data?.sale;
      companyInformation = model.data?.companyInformation;
      await printOvoReceipt(saleDataModel);
      Get.back();
      Get.back();
      Get.back();
    } else {
      CustomSnackBar.error(
          errorList: model.message ?? [MyStrings.requestFail.tr]);
    }
  }

  int paymentMethodCurrentIndex = -1;
  void changeSelectedPaymentMethod(int index) {
    paymentMethodCurrentIndex = index;
    selectedPaymentMethod = paymentMethodList[index];

    selectedPaymentAccount = PaymentAccountModel();
    paymentAccountsDataList.clear();
    paymentAccountsDataList.insert(
        0, PaymentAccountModel(id: null, accountName: MyStrings.selectOne.tr));
    paymentAccountsDataList.addAll(selectedPaymentMethod.paymentAccounts ?? []);
    update();
  }

  void setSelectedPaymentAccount(value) {
    selectedPaymentAccount = value;
    update();
  }

  bool isConfirmLoading = false;
  bool isSaleAndPrintLoading = false;
  bool isPdfDownloadLoading = false;

  Future<void> initBluetooth() async {
    try {
      // Listen to connection state
      _stateSubscription = bluetoothPrintService.state.listen((state) {
        switch (state) {
          case BluetoothPrintService.CONNECTED:
            connected = true;
            statusMessage = 'Connected success';
            update();
            break;
          case BluetoothPrintService.DISCONNECTED:
            connected = false;
            statusMessage = 'Disconnected success';
            update();
            break;
          default:
            break;
        }
      });

      // Listen to scanning state
      _scanningSubscription =
          bluetoothPrintService.isScanning.listen((isScanning) {
        // Handle scanning state changes if needed
        update();
      });

      // Check if already connected
      bool? isConnected = await bluetoothPrintService.isConnected;
      if (isConnected == true) {
        connected = true;
        statusMessage = 'Already connected';
        update();
      }

      // Start initial scan
      try {
        await bluetoothPrintService.safeScan(timeout: Duration(seconds: 4));
      } catch (e) {
        printE('Initial scan failed: $e');
      }

      printE(statusMessage);
      printE(connected);
    } catch (e) {
      statusMessage = 'Bluetooth initialization failed: $e';
      printE('Bluetooth init error: $e');
      update();
    }
  }

  void connectDevice({bool isPrintable = true}) async {
    try {
      if (selectedDevice != null && selectedDevice?.address != null) {
        statusMessage = 'connecting...';
        update();

        await bluetoothPrintService.connect(selectedDevice!);

        Get.back();

        if (isPrintable) {
          Future.delayed(Duration(seconds: 3), () async {
            await onConfirmSale(isSaleAndPrint: true);
          });
        } else {
          CustomSnackBar.success(
              successList: ["Connected to ${selectedDevice?.name ?? ""} "]);
        }
      } else {
        statusMessage = 'please select device';
        update();
      }
    } catch (e) {
      statusMessage = 'Connection failed: $e';
      update();
      printE('Connect error: $e');
    }
  }

  void disconnectDevice() async {
    if (connected) {
      statusMessage = 'disconnecting...';
      update();

      try {
        await bluetoothPrintService.disconnect();
        connected = false;
        statusMessage = 'Disconnected';
        selectedDevice = null;
        update();
      } catch (e) {
        statusMessage = 'Disconnect failed: $e';
        update();
        printE('Disconnect error: $e');
      }
    }
  }

  Future<void> printReceiptDemo() async {
    if (!connected) {
      CustomSnackBar.error(errorList: ["No device connected"]);
      return;
    }

    try {
      Map<String, dynamic> config = {
        'paperWidth': 58,
        'codePage': 'CP437',
      };

      List<LineText> list = [
        LineText(
          type: LineText.TYPE_TEXT,
          content: '**********************************************',
          size: 1,
          align: 1, // Center align
        ),
        LineText(
          type: LineText.TYPE_TEXT,
          content: 'DEMO TEXT',
          size: 2,
          align: 1, // Center align
        ),
        LineText(
          type: LineText.TYPE_TEXT,
          content: '',
        ),
      ];

      bool result = await bluetoothPrintService.printReceipt(config, list);

      if (result) {
        CustomSnackBar.success(successList: ["Demo print successful"]);
      } else {
        CustomSnackBar.error(errorList: ["Demo print failed"]);
      }
    } catch (e) {
      CustomSnackBar.error(errorList: ["Print error: $e"]);
      printE('Print demo error: $e');
    }
  }

  Future<bool> checkBluetoothStatus() async {
    try {
      // Check if Bluetooth is on using bluetooth_print_plus
      bool isOn = await bluetoothPrintService.isOn;

      if (!isOn) {
        CustomSnackBar.error(errorList: [
          "This app needs Bluetooth permissions to function properly. Please enable them in settings"
        ]);
        return false;
      }
      return true;
    } catch (e) {
      printE('Bluetooth status check error: $e');
      CustomSnackBar.error(errorList: ["Failed to check Bluetooth status"]);
      return false;
    }
  }

  printOvoReceipt(prefix.SaleDataModel? sale) async {
    if (sale == null) {
      printX("No sale data available.");
      return;
    }

    if (!connected) {
      CustomSnackBar.error(errorList: ["No printer connected"]);
      return;
    }

    try {
      Map<String, dynamic> config = {
        'paperWidth': 58,
        'codePage': 'CP437',
      };

      List<LineText> list = [];

      // Title
      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: companyInformation?.name ?? "",
        size: 2,
        align: 1, // Center align
      ));

      if (companyInformation?.address != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: companyInformation?.address ?? "",
          align: 1, // Center align
        ));
      }

      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: "${MyStrings.email}:${companyInformation?.email ?? ""}",
        align: 1, // Center align
      ));

      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: "${MyStrings.phone}:${companyInformation?.phone ?? ""}",
        align: 1, // Center align
      ));

      // Invoice Number
      if (sale.invoiceNumber != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: 'Invoice: ${sale.invoiceNumber}',
          size: 2,
          align: 1, // Center align
        ));
      }

      if (sale.saleDate != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: 'Date: ${sale.saleDate}',
          align: 1, // Center align
        ));
      }

      // Customer Information
      if (sale.customer?.name != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: MyStrings.billTo.tr,
          size: 2,
          align: 0, // Left align
        ));

        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: '${MyStrings.name.tr}: ${sale.customer!.name}',
          align: 0, // Left align
        ));
      }

      if (sale.customer?.email != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: '${MyStrings.email.tr}: ${sale.customer!.email}',
          align: 0, // Left align
        ));
      }

      if (sale.customer?.mobile != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: 'Phone: ${sale.customer!.mobile}',
          align: 0, // Left align
        ));
      }

      // Products section
      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: '--- Products ---',
        align: 1, // Center align
      ));

      if (sale.saleDetails != null && sale.saleDetails!.isNotEmpty) {
        for (var product in sale.saleDetails!) {
          String productName = product.productDetail?.product?.name ?? 'N/A';
          String quantity = product.quantity ?? "";
          String price = AppConverter.formatNumber(
              product.productDetail?.finalPrice ?? "");
          String total = AppConverter.formatNumber(product.subtotal ?? "");

          list.add(LineText(
            type: LineText.TYPE_TEXT,
            content: productName,
            align: 0, // Left align
          ));

          list.add(LineText(
            type: LineText.TYPE_TEXT,
            content: '$quantity x $price = $total $defaultCurrency',
            align: 0, // Left align
          ));
        }
      } else {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: 'No Products Found',
          align: 0, // Left align
        ));
      }

      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: '---------------------------------',
        align: 1, // Center align
      ));

      // Totals
      if (sale.subtotal != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content:
              'Subtotal: ${AppConverter.formatNumber(sale.subtotal.toString())} $defaultCurrency',
          align: 2, // Right align
        ));
      }

      if (sale.discountAmount != null && sale.discountAmount != "0") {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content:
              'Discount: ${AppConverter.formatNumber(sale.discountAmount.toString())} $defaultCurrency',
          align: 2, // Right align
        ));
      }

      if (sale.shippingAmount != null && sale.shippingAmount != "0") {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content:
              'Shipping: ${AppConverter.formatNumber(sale.shippingAmount.toString())} $defaultCurrency',
          align: 2, // Right align
        ));
      }

      if (sale.total != null) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content:
              'Total: ${AppConverter.formatNumber(sale.total.toString())} $defaultCurrency',
          align: 2, // Right align
        ));
      }

      if (selectedPaymentMethod.name.toString().trim().isNotEmpty) {
        list.add(LineText(
          type: LineText.TYPE_TEXT,
          content: 'Method: ${selectedPaymentMethod.name ?? ""}',
          align: 2, // Right align
        ));
      }

      // Footer
      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: 'Thank you for your purchase!',
        align: 1, // Center align
      ));

      list.add(LineText(
        type: LineText.TYPE_TEXT,
        content: '',
        align: 1, // Center align
      ));

      // Print the receipt
      bool result = await bluetoothPrintService.printReceipt(config, list);

      if (result) {
        CustomSnackBar.success(successList: ["Receipt printed successfully"]);
      } else {
        CustomSnackBar.error(errorList: ["Failed to print receipt"]);
      }
    } catch (e) {
      CustomSnackBar.error(errorList: ["Print error: $e"]);
      printE('Print receipt error: $e');
    }
  }

  onConfirmSale(
      {bool isSaleAndSavePdf = false, bool isSaleAndPrint = false}) async {
    if (selectedPaymentMethod.id == null) {
      CustomSnackBar.error(errorList: ["Select payment method"]);
      return;
    }

    List<SaleProduct> checkoutList =
        Get.find<CheckoutController>().checkoutList;

    Map<String, dynamic> map = {
      "sale_date": DateConverter.formatDateOnly(DateTime.now()),
      "customer_id": customerData.id ?? "1",
      "warehouse_id":
          Get.find<GlobalController>().selectedWareHouse.id.toString(),
      "status": "1",
      "discount_type":
          billingController.selectedDiscountType == MyStrings.percent
              ? "1"
              : "2",
      "discount_value": billingController.discountAmount.toString(),
      "shipping_amount": billingController.deliveryCharge.toString(),
      "paying_amount": billingController.getTotal(),
      "note": noteController.text,
      "coupon_id": billingController.couponId,
      "save_action_type": isSaleAndSavePdf ? "save_and_print" : "only_save",
    };

    for (int index = 0; index < checkoutList.length; index++) {
      map["sale_details[$index][product_id]"] =
          checkoutList[index].original?.productId.toString() ?? "";
      map["sale_details[$index][product_detail_id]"] =
          checkoutList[index].id.toString();
      map["sale_details[$index][quantity]"] =
          checkoutList[index].finalQuantity.toString();
    }

    map["payment[0][amount]"] = billingController.getTotal();
    map["payment[0][payment_type]"] = selectedPaymentMethod.id.toString();
    map["payment[0][payment_account_id]"] =
        selectedPaymentAccount.id.toString();

    if (isSaleAndPrint) {
      isSaleAndPrintLoading = true;
    } else if (isSaleAndSavePdf) {
      isPdfDownloadLoading = true;
    } else {
      isConfirmLoading = true;
    }
    update();
    ResponseModel responseModel = await paymentRepo.confirmSale(map: map);

    if (responseModel.statusCode == 200) {
      ConfirmSaleResponseModel model =
          ConfirmSaleResponseModel.fromJson(responseModel.responseJson);

      if (model.status.toString() ==
          MyStrings.success.toString().toLowerCase()) {
        if (isSaleAndSavePdf) {
          saleId = model.data?.sale?.id.toString() ?? "";
          String url =
              "${UrlContainer.baseUrl}${UrlContainer.downloadPdfEndPoint}/$saleId";
          CustomSnackBar.success(successList: model.message ?? []);
          CustomSnackBar.success(successList: ["Invoice is downloading..."]);

          AnyFileDownloaderAndOpener pdfDownloader =
              AnyFileDownloaderAndOpener();
          await pdfDownloader.downloadAnyFile(url: url).then((value) {
            update();
          });

          isPdfDownloadLoading = false;
          Get.back();
          Get.back();
          Get.back();
        }

        if (isSaleAndPrint) {
          saleId = model.data?.sale?.id.toString() ?? "";
          await viewSaleData(model: model);
          isSaleAndPrintLoading = false;
          update();
        } else {
          Get.back();
          Get.back();
          Get.back();
          CustomSnackBar.success(successList: model.message ?? []);
        }
      } else {
        CustomSnackBar.error(
            errorList: model.message ?? [MyStrings.requestFail.tr]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message.tr]);
    }

    isConfirmLoading = false;
    isSaleAndPrintLoading = false;
    isPdfDownloadLoading = false;
    update();
  }

  clearOldData() {
    noteController.clear();
    customerData = CustomerData();
    selectedPaymentMethod = PaymentMethod();
  }

  // Additional helper methods for the new service
  Future<List<BluetoothDevice>> getAvailableDevices() async {
    try {
      return await bluetoothPrintService.safeScan(
          timeout: Duration(seconds: 10));
    } catch (e) {
      printE('Get devices error: $e');
      return [];
    }
  }

  Stream<List<BluetoothDevice>> get scanResults =>
      bluetoothPrintService.scanResults;
  Stream<bool> get isScanning => bluetoothPrintService.isScanning;

  Future<void> startScan() async {
    try {
      await bluetoothPrintService.safeScan(timeout: Duration(seconds: 10));
    } catch (e) {
      statusMessage = 'Scan failed: $e';
      update();
      printE('Start scan error: $e');
    }
  }

  Future<void> stopScan() async {
    try {
      await bluetoothPrintService.stopScan();
    } catch (e) {
      printE('Stop scan error: $e');
    }
  }

  void selectDevice(BluetoothDevice device) {
    selectedDevice = device;
    update();
  }
}
