import 'dart:async';
import 'package:get/get.dart';
import 'package:ovosale/core/helper/string_format_helper.dart';
import 'package:ovosale/core/utils/util.dart';
import 'package:ovosale/data/model/checkout/checkout_cart_response_model.dart';
import 'package:ovosale/data/repo/checkout/checkout_repo.dart';
import '../../../app/components/snack_bar/show_custom_snackbar.dart';
import '../../../core/utils/my_strings.dart';
import '../../model/authorization/authorization_response_model.dart';
import '../../model/global/response_model/response_model.dart';
import '../../model/new_sale/new_sale_response_model.dart';

class CheckoutController extends GetxController {
  CheckoutRepo checkoutRepo;
  CheckoutController({required this.checkoutRepo});

  bool isLoading = true;

  List<SaleProduct> checkoutList = [];

  Future<void> loadCheckoutData() async {
    ResponseModel responseModel = await checkoutRepo.getCheckoutData();

    if (responseModel.statusCode == 200) {
      CheckoutCartResponseModel model =
          CheckoutCartResponseModel.fromJson(responseModel.responseJson);

      if (model.status.toString() ==
          MyStrings.success.toString().toLowerCase()) {
        checkoutList.clear();

        checkoutList.addAll(model.data?.products ?? []);

        for (var product in checkoutList) {
          product.quantityController?.text = product.quantity ?? "0";
          product.finalQuantity = AppConverter.toInt(product.quantity);
        }
      } else {
        CustomSnackBar.error(
            errorList: model.message ?? [MyStrings.requestFail.tr]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message.tr]);
    }

    isLoading = false;
    update();
  }

  bool isUpdateQuantity = false;
  void updateQuantity(
      {required String quantity, required String cartId}) async {
    isUpdateQuantity = true;
    update();

    ResponseModel responseModel =
        await checkoutRepo.updateQuantity(quantity: quantity, cardId: cartId);

    if (responseModel.statusCode == 200) {
      AuthorizationResponseModel model =
          AuthorizationResponseModel.fromJson(responseModel.responseJson);

      if (model.status.toString() ==
          MyStrings.success.toString().toLowerCase()) {
        await loadCheckoutData();
      } else {
        CustomSnackBar.error(
            errorList: model.message ?? [MyStrings.requestFail.tr]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message.tr]);
    }

    isUpdateQuantity = false;
    update();
  }

  bool isRemoveCartLoading = false;
  String isRemoveCartLoadingID = "-1";
  void removeCart({required String cartId}) async {
    isRemoveCartLoading = true;
    isRemoveCartLoadingID = cartId;
    update();

    try {
      ResponseModel responseModel =
          await checkoutRepo.removeCart(cartId: cartId);

      if (responseModel.statusCode == 200) {
        AuthorizationResponseModel model =
            AuthorizationResponseModel.fromJson(responseModel.responseJson);

        if (model.status.toString() ==
            MyStrings.success.toString().toLowerCase()) {
          await loadCheckoutData();
        } else {
          CustomSnackBar.error(
              errorList: model.message ?? [MyStrings.requestFail.tr]);
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message.tr]);
      }
    } catch (e) {
      printE(e);
    }

    isRemoveCartLoading = false;
    isRemoveCartLoadingID = "-1";
    update();
  }

  bool isClearCartLoading = false;
  void clearCart() async {
    isClearCartLoading = true;
    update();

    Get.back();

    ResponseModel responseModel = await checkoutRepo.clearCart();

    if (responseModel.statusCode == 200) {
      AuthorizationResponseModel model =
          AuthorizationResponseModel.fromJson(responseModel.responseJson);

      if (model.status.toString() ==
          MyStrings.success.toString().toLowerCase()) {
        await loadCheckoutData();
      } else {
        CustomSnackBar.error(
            errorList: model.message ?? [MyStrings.requestFail.tr]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message.tr]);
    }

    isClearCartLoading = false;
    update();
  }

  String getSubTotal({required SaleProduct saleProduct}) {
    double price = AppConverter.toDouble(saleProduct.price);
    int quantity = AppConverter.toInt(saleProduct.quantity);

    return AppConverter.formatNumber("${price * quantity}");
  }

  String getTotalPayableAmount() {
    double totalPayableAmount = 0.00;
    for (var product in checkoutList) {
      double subTotal =
          AppConverter.toDouble(getSubTotal(saleProduct: product));
      totalPayableAmount += subTotal;
    }
    return AppConverter.formatNumber("$totalPayableAmount");
  }

  Timer? timer;
  void startIncreasing({required SaleProduct product}) {
    timer = Timer.periodic(Duration(milliseconds: 100), (timer) {
      int stock = AppConverter.toInt(product.inStock);

      if (product.finalQuantity < stock) {
        product.finalQuantity++;
      }

      update();
    });
  }

  void startDecreasing({required SaleProduct product}) {
    timer = Timer.periodic(Duration(milliseconds: 100), (timer) {
      if (product.finalQuantity > 1) {
        product.finalQuantity--;
      }
      update();
    });
  }

  void stopTimer() {
    timer?.cancel();
  }
}
