import 'dart:async';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/divider/custom_divider.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/customer_database_Shimmer.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/screens/supplier/widgets/supplier_list_tile.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/supplier/supplier_controller.dart';
import 'package:ovosale/data/repo/supplier/supplier_repo.dart';

import '../../../core/app_permission/app_permission.dart';
import '../../../data/controller/cart/global_controller.dart';

class SupplierScreen extends StatefulWidget {
  const SupplierScreen({super.key});

  @override
  State<SupplierScreen> createState() => _SupplierScreenState();
}

class _SupplierScreenState extends State<SupplierScreen> {
  final ScrollController scrollController = ScrollController();
  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<SupplierController>().hasNext()) {
        Get.find<SupplierController>().getSupplierData(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;
  @override
  void initState() {
    Get.put(SupplierRepo());
    final controller = Get.put(SupplierController(supplierRepo: Get.find()));
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      scrollController.addListener(scrollListener);
      controller.getSupplierData();
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<SupplierController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.supplier.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            Container(
              padding: EdgeInsetsDirectional.symmetric(
                  horizontal: Dimensions.space16.w,
                  vertical: Dimensions.space12.h),
              color: MyColor.white,
              child: LabelTextField(
                labelText: "",
                hideLabel: true,
                hintText: MyStrings.search.tr,
                controller: controller.searchSupplierController,
                hintTextColor: MyColor.getBodyTextColor(),
                onChanged: (value) {
                  if (_debounceTimer?.isActive ?? false) {
                    _debounceTimer?.cancel();
                  }

                  _debounceTimer = Timer(const Duration(milliseconds: 500), () {
                    controller.getSupplierData();
                  });
                },
                textInputType: TextInputType.text,
                textInputAction: TextInputAction.done,
                radius: Dimensions.largeRadius,
                validator: (value) {
                  if (value!.isEmpty) {
                    return MyStrings.fieldErrorMsg.tr;
                  } else {
                    return null;
                  }
                },
                prefixIcon: Icon(
                  CupertinoIcons.search,
                  size: Dimensions.textFieldIconSize,
                ),
              ),
            ),
            spaceDown(Dimensions.space10),
            controller.isLoading
                ? CustomerDatabaseShimmer()
                : controller.filteredSupplierData.isEmpty
                    ? Center(
                        child: SizedBox(
                          height: MediaQuery.of(context).size.height * 0.5,
                          child: NoDataWidget(),
                        ),
                      )
                    : Expanded(
                        child: CustomAppCard(
                            radius: 0,
                            showBorder: false,
                            child: ListView.separated(
                                controller: scrollController,
                                separatorBuilder: (context, index) =>
                                    CustomDivider(space: Dimensions.space16.h),
                                shrinkWrap: true,
                                itemCount:
                                    controller.filteredSupplierData.length + 1,
                                itemBuilder: (context, i) {
                                  if (i ==
                                      controller.filteredSupplierData.length) {
                                    return controller.hasNext()
                                        ? SizedBox(
                                            height: Dimensions.space40,
                                            width: MediaQuery.of(context)
                                                .size
                                                .width,
                                            child: const CustomLoader(
                                                isPagination: true),
                                          )
                                        : const SizedBox();
                                  }
                                  return SupplierListTile(
                                    index: i,
                                    suppliersData:
                                        controller.filteredSupplierData,
                                  );
                                })),
                      )
          ],
        ),
        floatingActionButton: CustomAppCard(
            onPressed: () {
              if (Get.find<GlobalController>()
                  .checkPermission(featureName: AppPermissions.addSupplier)) {
                Get.toNamed(RouteHelper.newSupplierScreen)?.then((v) {
                  controller.getSupplierData();
                });
              }
            },
            height: Dimensions.space64.h,
            width: Dimensions.space64.w,
            padding: EdgeInsetsDirectional.all(Dimensions.space20),
            backgroundColor: MyColor.getPrimaryColor(),
            showBorder: false,
            child: MyAssetImageWidget(
              color: theme.cardColor,
              assetPath: MyIcons.add,
              isSvg: true,
            )),
      ),
    );
  }
}
