import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/customer_database_shimmer.dart';
import 'package:ovosale/app/screens/sale_list/widget/menu_Item_widget.dart';
import 'package:ovosale/app/screens/sale_list/widget/right_top_sale.dart';
import 'package:ovosale/app/screens/sale_list/widget/sale_status.dart';
import 'package:ovosale/core/app_permission/app_permission.dart';
import 'package:ovosale/data/controller/sale_list/sale_list_controller.dart';
import 'package:ovosale/data/repo/sale_list/sale_list_repo.dart';

import '../../../core/helper/string_format_helper.dart';
import '../../../core/utils/dimensions.dart';
import '../../../core/utils/my_color.dart';
import '../../../core/utils/my_icons.dart';
import '../../../core/utils/my_images.dart';
import '../../../core/utils/my_strings.dart';
import '../../../core/utils/url_container.dart';
import '../../../data/controller/file_download/file_downloader.dart';
import '../../../data/services/shared_pref_service.dart';
import '../../components/card/custom_app_card.dart';
import '../../components/custom_loader/custom_loader.dart';
import '../../components/image/my_asset_widget.dart';
import '../../components/text-field/label_text_field.dart';
import '../report/widget/card_elements.dart';

class SaleListScreen extends StatefulWidget {
  const SaleListScreen({super.key});

  @override
  State<SaleListScreen> createState() => _SaleListScreenState();
}

class _SaleListScreenState extends State<SaleListScreen> {
  final ScrollController scrollController = ScrollController();
  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<SaleListController>().hasNext()) {
        Get.find<SaleListController>().getSalesListData(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;

  @override
  void initState() {
    Get.put(SaleListRepo());
    final controller = Get.put(SaleListController(saleListRepo: Get.find()));

    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.getSalesListData();
      scrollController.addListener(scrollListener);
    });
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);

    return GetBuilder<SaleListController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.sale.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            SizedBox(height: 12.h),
            Padding(
              padding: const EdgeInsets.symmetric(horizontal: 10),
              child: Row(
                children: [
                  Expanded(
                    child: LabelTextField(
                        controller: controller.searchController,
                        contentPadding: EdgeInsetsDirectional.symmetric(
                            vertical: Dimensions.space14.h,
                            horizontal: Dimensions.space16.w),
                        labelText: "",
                        hideLabel: true,
                        hintText: MyStrings.search.tr,
                        hintTextColor: MyColor.getBodyTextColor(),
                        onChanged: (value) {
                          if (_debounceTimer?.isActive ?? false) {
                            _debounceTimer?.cancel();
                          }

                          // Start a new timer
                          _debounceTimer =
                              Timer(const Duration(milliseconds: 500), () {
                            controller.onSearch();
                          });
                        },
                        textInputType: TextInputType.text,
                        textInputAction: TextInputAction.done,
                        radius: Dimensions.largeRadius,
                        validator: (value) {
                          if (value!.isEmpty) {
                            return MyStrings.fieldErrorMsg.tr;
                          } else {
                            return null;
                          }
                        },
                        prefixIcon: Padding(
                          padding: const EdgeInsets.all(13),
                          child: MyAssetImageWidget(
                            assetPath: MyImages.search,
                            isSvg: true,
                            width: 16,
                            height: 16,
                          ),
                        ),
                        suffixIcon: IconButton(
                            onPressed: () {
                              controller.searchController.clear();
                              controller.getSalesListData();
                              controller.update();
                            },
                            icon: Icon(
                              Icons.close,
                              size: 20,
                              color: MyColor.getBodyTextColor(),
                            ))),
                  ),
                ],
              ),
            ),
            SizedBox(height: 12.h),
            Expanded(
                child: controller.isLoading
                    ? CustomerDatabaseShimmer()
                    : controller.saleList.isEmpty && !controller.isLoading
                        ? Center(
                            child: SizedBox(
                              height: MediaQuery.of(context).size.height * 0.5,
                              child: NoDataWidget(),
                            ),
                          )
                        : ListView.builder(
                            controller: scrollController,
                            itemCount: controller.saleList.length + 1,
                            itemBuilder: (context, index) {
                              if (controller.saleList.length == index) {
                                return controller.hasNext()
                                    ? const CustomLoader(isPagination: true)
                                    : const SizedBox();
                              }

                              var sales = controller.saleList[index];

                              return CustomAppCard(
                                  padding: EdgeInsets.symmetric(
                                      horizontal: 10.h, vertical: 8.h),
                                  margin: EdgeInsets.symmetric(
                                      horizontal: 10.h, vertical: 5.h),
                                  radius: 8,
                                  child: Column(
                                    children: [
                                      Row(
                                        mainAxisAlignment:
                                            MainAxisAlignment.spaceBetween,
                                        children: [
                                          LeftTopSection(
                                            title:
                                                "${MyStrings.invoice.tr}/${MyStrings.totalItem.tr}",
                                            value1: sales.invoiceNumber,
                                            value2:
                                                "${sales.saleDetailsCount} ${MyStrings.items.tr}",
                                          ),
                                          CenterSection(
                                            title:
                                                "${MyStrings.total.tr} | ${MyStrings.paidAmount.tr}",
                                            value1:
                                                "${SharedPreferenceService.getCurrencySymbol()}${AppConverter.formatNumber(sales.total.toString())}",
                                            value2:
                                                "${SharedPreferenceService.getCurrencySymbol()}${AppConverter.formatNumber(sales.paymentsSumAmount ?? "0.00")}",
                                          ),
                                          RightTopSaleSection(
                                            title: MyStrings.status.tr,
                                            statusWidget: SaleStatusWidget(
                                                status: sales.status ?? "-1"),
                                          )
                                        ],
                                      ),
                                      SizedBox(height: 12),
                                      Row(
                                        mainAxisAlignment:
                                            MainAxisAlignment.spaceBetween,
                                        children: [
                                          BottomLeftSection(
                                            title: MyStrings.customer.tr,
                                            value1: sales.customer?.name ?? "",
                                          ),
                                          BottomCenterSection(
                                            title: MyStrings.saleDate.tr,
                                            value1: sales.saleDate ?? "",
                                          ),
                                          Expanded(
                                            child: Row(
                                              mainAxisAlignment:
                                                  MainAxisAlignment.end,
                                              children: [
                                                PopupMenuButton<String>(
                                                  color: MyColor.white,
                                                  padding: EdgeInsets.zero,
                                                  child: CustomAppCard(
                                                    radius: Dimensions.space8,
                                                    borderColor: MyColor
                                                        .lightInformation
                                                        .withValues(alpha: .3),
                                                    // padding: EdgeInsetsDirectional.all(Dimensions.space8),
                                                    padding: const EdgeInsets
                                                        .symmetric(
                                                        horizontal: 12,
                                                        vertical: 4),
                                                    backgroundColor:
                                                        MyColor.transparent,
                                                    width: 85.w,
                                                    child: sales.isLoading
                                                        ? CustomLoader()
                                                        : Row(
                                                            mainAxisAlignment:
                                                                MainAxisAlignment
                                                                    .spaceBetween,
                                                            children: [
                                                              Text(
                                                                MyStrings
                                                                    .action.tr,
                                                                style: theme
                                                                    .textTheme
                                                                    .titleSmall
                                                                    ?.copyWith(
                                                                        fontSize:
                                                                            12,
                                                                        color: MyColor
                                                                            .lightInformation),
                                                              ),
                                                              MyAssetImageWidget(
                                                                assetPath: MyIcons
                                                                    .arrowDown,
                                                                isSvg: true,
                                                                height:
                                                                    Dimensions
                                                                        .space12
                                                                        .h,
                                                                width:
                                                                    Dimensions
                                                                        .space12
                                                                        .w,
                                                                color: MyColor
                                                                    .lightInformation,
                                                              )
                                                            ],
                                                          ),
                                                  ),
                                                  onSelected: (value) {},
                                                  itemBuilder:
                                                      (BuildContext context) {
                                                    return [
                                                      PopupMenuItem<String>(
                                                          onTap: () async {
                                                            AnyFileDownloaderAndOpener
                                                                pdfDownloader =
                                                                AnyFileDownloaderAndOpener();

                                                            String url =
                                                                "${UrlContainer.baseUrl}${UrlContainer.downloadPdfEndPoint}/${sales.id}";

                                                            sales.isLoading =
                                                                true;
                                                            controller.update();
                                                            await pdfDownloader
                                                                .downloadAnyFile(
                                                                    url: url,
                                                                    fileName: sales
                                                                        .invoiceNumber)
                                                                .then((value) {
                                                              sales.isLoading =
                                                                  false;
                                                              controller
                                                                  .update();
                                                            });
                                                          },
                                                          value: MyStrings
                                                              .viewInvoice,
                                                          child: MenuItemWidget(
                                                              title: MyStrings
                                                                  .viewInvoice)),
                                                      PopupMenuItem<String>(
                                                          onTap: () async {
                                                            if (controller
                                                                .globalController
                                                                .checkPermission(
                                                                    featureName:
                                                                        AppPermissions
                                                                            .downloadSaleInvoice)) {
                                                              AnyFileDownloaderAndOpener
                                                                  pdfDownloader =
                                                                  AnyFileDownloaderAndOpener();

                                                              String url =
                                                                  "${UrlContainer.baseUrl}${UrlContainer.downloadPdfEndPoint}/${sales.id}";

                                                              sales.isLoading =
                                                                  true;
                                                              controller
                                                                  .update();
                                                              await pdfDownloader
                                                                  .downloadAnyFile(
                                                                      url: url,
                                                                      fileName:
                                                                          "${sales.invoiceNumber}.pdf",
                                                                      isView:
                                                                          false)
                                                                  .then(
                                                                      (value) {
                                                                sales.isLoading =
                                                                    false;
                                                                controller
                                                                    .update();
                                                              });
                                                            }
                                                          },
                                                          value: MyStrings
                                                              .downloadInvoice,
                                                          child: MenuItemWidget(
                                                              title: MyStrings
                                                                  .downloadInvoice)),
                                                    ];
                                                  },
                                                ),
                                              ],
                                            ),
                                          ),
                                        ],
                                      )
                                    ],
                                  ));
                            },
                          ))
          ],
        ),
      ),
    );
  }
}
