import 'package:flutter/material.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/shimmer/stock_report_shimmer.dart';
import 'package:ovosale/app/screens/report/widget/warehouse_section.dart';
import 'package:ovosale/core/utils/dimensions.dart';

import '../../../../core/utils/my_color.dart';
import '../../../../data/controller/report/report_controller.dart';
import '../../../../data/repo/report_repo/report_repo.dart';
import '../../../components/card/custom_app_card.dart';
import '../../../components/custom_loader/custom_loader.dart';
import '../../../components/image/my_network_image_widget.dart';
import '../../../components/no_data.dart';

class StockWidget extends StatefulWidget {
  const StockWidget({
    super.key,
  });

  @override
  State<StockWidget> createState() => _StockWidgetState();
}

class _StockWidgetState extends State<StockWidget> {
  final ScrollController scrollController = ScrollController();
  void scrollListener() {
    if (scrollController.position.pixels == scrollController.position.maxScrollExtent) {
      if (Get.find<ReportController>().hasNext()) {
        Get.find<ReportController>().getStockData(isPagination: true);
      }
    }
  }

  @override
  void initState() {
    Get.put(ReportRepo());
    final controller = Get.put(ReportController(reportRepo: Get.find()));

    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.getStockData();
      scrollController.addListener(scrollListener);
    });
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);

    return GetBuilder<ReportController>(
      builder: (controller) => controller.isLoading
          ? StockReportShimmer()
          : controller.stockList.isEmpty
              ? NoDataWidget()
              : Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    GestureDetector(
                      onTap: () {
                        showModalBottomSheet(
                            shape: RoundedRectangleBorder(borderRadius: BorderRadius.only(topLeft: Radius.circular(15), topRight: Radius.circular(15))),
                            isScrollControlled: true,
                            useSafeArea: true,
                            context: context,
                            builder: (context) {
                              return StockWarehouseSection();
                            });
                      },
                      child: Align(
                        alignment: Alignment.centerRight,
                        child: CustomAppCard(
                            margin: EdgeInsetsDirectional.only(end: 12, bottom: 4),
                            padding: EdgeInsets.symmetric(horizontal: 12, vertical: 8),
                            radius: 8,
                            child: Row(
                              mainAxisSize: MainAxisSize.min,
                              children: [
                                Icon(
                                  Icons.place_rounded,
                                  color: theme.primaryColor,
                                  size: 16,
                                ),
                                SizedBox(width: 4.w),
                                Text(
                                  controller.globalCartCountController.selectedWareHouse.name ?? "",
                                  style: theme.textTheme.titleSmall?.copyWith(fontSize: 12),
                                )
                              ],
                            )),
                      ),
                    ),
                    Expanded(
                      child: ListView.builder(
                        padding: EdgeInsets.only(bottom: Dimensions.space100.h),
                        controller: scrollController,
                        itemCount: controller.stockList.length + 1,
                        itemBuilder: (context, index) {
                          if (controller.stockList.length == index) {
                            return controller.hasNext() ? const CustomLoader(isPagination: true) : const SizedBox();
                          }

                          var stock = controller.stockList[index];

                          return CustomAppCard(
                              padding: EdgeInsets.symmetric(horizontal: 8.h, vertical: 8.h),
                              margin: EdgeInsets.symmetric(horizontal: 10.h, vertical: 4.h),
                              radius: 8,
                              child: Row(
                                children: [
                                  MyNetworkImageWidget(
                                    imageUrl: stock.product?.imageSrc ?? "",
                                    width: 50,
                                    height: 50,
                                    radius: 100,
                                  ),
                                  SizedBox(width: 8.w),
                                  Expanded(
                                    child: Column(
                                      crossAxisAlignment: CrossAxisAlignment.start,
                                      children: [
                                        Text(
                                          "${stock.product?.name}",
                                          style: theme.textTheme.titleSmall,
                                        ),
                                        SizedBox(
                                          height: 2,
                                        ),
                                        Text(
                                          "${stock.sku}",
                                          style: theme.textTheme.bodyMedium,
                                        ),
                                      ],
                                    ),
                                  ),
                                  CustomAppCard(alignment: Alignment.center, backgroundColor: MyColor.getSuccessColor().withValues(alpha: .05), padding: EdgeInsets.symmetric(horizontal: 8, vertical: 2), radius: 4, borderColor: MyColor.getSuccessColor().withValues(alpha: .05), child: Text(stock.productStockSumStock ?? "0", style: theme.textTheme.titleSmall?.copyWith(color: MyColor.getSuccessColor(), fontSize: 14))),
                                ],
                              ));
                        },
                      ),
                    ),
                  ],
                ),
    );
  }
}
