import 'dart:async';
import 'package:flutter/material.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/screens/report/widget/expense_report_widget.dart';
import 'package:ovosale/app/screens/report/widget/profit_loss_report.dart';
import 'package:ovosale/app/screens/report/widget/purchase_report_widget.dart';
import 'package:ovosale/app/screens/report/widget/sales_report_widget.dart';
import 'package:ovosale/app/screens/report/widget/stock_report_widget.dart';
import 'package:ovosale/core/utils/my_color.dart';
import 'package:ovosale/data/controller/report/report_controller.dart';
import 'package:ovosale/data/repo/report_repo/report_repo.dart';
import '../../../core/utils/dimensions.dart';
import '../../../core/utils/my_images.dart';
import '../../../core/utils/my_strings.dart';
import '../../components/image/my_asset_widget.dart';
import '../../components/text-field/label_text_field.dart';

class ReportScreen extends StatefulWidget {
  const ReportScreen({super.key});

  @override
  State<ReportScreen> createState() => _ReportScreenState();
}

class _ReportScreenState extends State<ReportScreen>
    with SingleTickerProviderStateMixin {
  late TabController _tabController;

  Timer? _debounceTimer;

  @override
  void initState() {
    Get.put(ReportRepo());
    Get.put(ReportController(reportRepo: Get.find()));

    super.initState();
    _tabController = TabController(length: 5, vsync: this, initialIndex: 0);

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {});
  }

  @override
  void dispose() {
    _tabController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);

    return GetBuilder<ReportController>(
      builder: (controller) => MyCustomScaffold(
          isShowLeading: false,
          pageTitle: MyStrings.report.tr,
          padding: EdgeInsets.zero,
          body: Column(
            children: [
              TabBar(
                controller: _tabController,
                isScrollable: true,
                indicatorColor: theme.primaryColor,
                labelColor: theme.primaryColor,
                unselectedLabelColor: Colors.grey,
                indicatorWeight: 3,
                indicatorPadding: EdgeInsets.zero,
                tabAlignment: TabAlignment.start,
                onTap: (value) {
                  controller.changeTab();
                },
                tabs: [
                  Tab(text: MyStrings.stockReport.tr),
                  Tab(text: MyStrings.salesReport.tr),
                  Tab(text: MyStrings.purchasesReport.tr),
                  Tab(text: MyStrings.expenseReport.tr),
                  Tab(text: MyStrings.profitLossReport.tr),
                ],
              ),
              SizedBox(height: Dimensions.space12.h),
              Padding(
                padding:
                    const EdgeInsets.symmetric(horizontal: Dimensions.space10),
                child: Row(
                  children: [
                    Expanded(
                      child: LabelTextField(
                          controller: controller.searchController,
                          contentPadding: EdgeInsetsDirectional.symmetric(
                              vertical: Dimensions.space14.h,
                              horizontal: Dimensions.space16.w),
                          labelText: "",
                          hideLabel: true,
                          hintText: MyStrings.search.tr,
                          hintTextColor: MyColor.getBodyTextColor(),
                          onChanged: (value) {
                            if (_debounceTimer?.isActive ?? false) {
                              _debounceTimer?.cancel();
                            }

                            // Start a new timer
                            _debounceTimer =
                                Timer(const Duration(milliseconds: 500), () {
                              controller.onSearch(_tabController.index, false);
                            });
                          },
                          textInputType: TextInputType.text,
                          textInputAction: TextInputAction.done,
                          radius: Dimensions.largeRadius,
                          validator: (value) {
                            if (value!.isEmpty) {
                              return MyStrings.fieldErrorMsg.tr;
                            } else {
                              return null;
                            }
                          },
                          prefixIcon: Padding(
                            padding: const EdgeInsets.all(13),
                            child: MyAssetImageWidget(
                              assetPath: MyImages.search,
                              isSvg: true,
                              width: 16,
                              height: 16,
                            ),
                          ),
                          suffixIcon:
                              controller.searchController.text.trim().isNotEmpty
                                  ? IconButton(
                                      onPressed: () {
                                        controller.onSearch(
                                            _tabController.index, true);
                                      },
                                      icon: Icon(
                                        Icons.close,
                                        size: 20,
                                        color: MyColor.getBodyTextColor(),
                                      ))
                                  : null),
                    ),
                  ],
                ),
              ),
              SizedBox(height: 12.h),
              Expanded(
                child: TabBarView(controller: _tabController, children: [
                  StockWidget(),
                  SalesReportWidget(),
                  PurchaseReportWidget(),
                  ExpenseReportWidget(space: 0),
                  ProfitLossReport(),
                ]),
              ),
            ],
          )),
    );
  }
}
