import 'dart:developer';
import 'dart:io';

import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:qr_code_scanner_plus/qr_code_scanner_plus.dart';

import '../../../data/controller/qr_code/qr_code_controller.dart';
import '../../../data/repo/qr_code/qr_code_repo.dart';

class QrCodeScannerScreen extends StatefulWidget {
  const QrCodeScannerScreen({super.key});

  @override
  State<QrCodeScannerScreen> createState() => _QrCodeScannerScreenState();
}

class _QrCodeScannerScreenState extends State<QrCodeScannerScreen> {
  final GlobalKey qrKey = GlobalKey(debugLabel: 'QR');

  Barcode? result;
  QRViewController? qrController;

  @override
  void initState() {
    Get.put(QrCodeRepo());
    Get.put(QrCodeController(qrCodeRepo: Get.find()));

    super.initState();
  }

  void _onQRViewCreated(QRViewController qrController) {
    this.qrController = qrController;
    qrController.scannedDataStream.listen((scanData) async {
      result = scanData;
      if (result?.code != null && result!.code!.isNotEmpty) {
        await manageBarCodeData(result!.code!);
      }
    });
  }

  Future manageBarCodeData(String barCode) async {
    final controller = Get.find<QrCodeController>();

    await controller.submitBarCodeData(barCode);
    try {
      qrController?.pauseCamera();
      // Navigator.pop(context);
    } catch (e) {
      printX(e);
      Navigator.pop(context);
    }
  }

  @override
  void reassemble() {
    if (Platform.isAndroid) {
      qrController!.pauseCamera();
    } else if (Platform.isIOS) {
      qrController!.resumeCamera();
    }
    super.reassemble();
  }

  @override
  void dispose() {
    super.dispose();
    qrController?.stopCamera();
    // qrController?.stopCamera();
  }

  void _onPermissionSet(BuildContext context, QRViewController ctrl, bool p) {
    log('${DateTime.now().toIso8601String()}_onPermissionSet $p');
    if (!p) {
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text(MyStrings.noPermission.tr)),
      );
    }
  }

  @override
  Widget build(BuildContext context) {
    if (qrController != null && mounted) {
      qrController!.pauseCamera();
      qrController!.resumeCamera();
    }

    return GetBuilder<QrCodeController>(
      builder: (viewController) => MyCustomScaffold(
        padding: EdgeInsetsDirectional.zero,
        pageTitle: MyStrings.scanBarCode.tr,
        body: viewController.isLoading
            ? CustomLoader()
            : Column(
                children: [
                  Expanded(
                      child: QRView(
                    key: qrKey,
                    onQRViewCreated: _onQRViewCreated,
                    cameraFacing: CameraFacing.back,
                    overlay: QrScannerOverlayShape(
                        borderRadius: 5,
                        borderLength: 30,
                        borderWidth: 10,
                        cutOutHeight: 170,
                        cutOutWidth: context.width * .75),
                    onPermissionSet: (ctrl, p) =>
                        _onPermissionSet(context, ctrl, p),
                  )),
                ],
              ),
      ),
    );
  }
}
