import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/text-field/custom_drop_down_text_field.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/core/helper/string_format_helper.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/new_product/new_product_controller.dart';
import 'package:ovosale/data/services/shared_pref_service.dart';

class ProductUpdateSecondStep extends StatefulWidget {
  final int index;

  const ProductUpdateSecondStep({super.key, required this.index});

  @override
  State<ProductUpdateSecondStep> createState() =>
      _ProductUpdateSecondStepState();
}

class _ProductUpdateSecondStepState extends State<ProductUpdateSecondStep> {
  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<NewProductController>(builder: (controller) {
      var item = controller.formData[widget.index];
      return Column(
        children: [
          Row(
            children: [
              Flexible(
                flex: 1,
                child: LabelTextField(
                  controller: item.basePriceController,
                  labelText: MyStrings.basePrice.tr,
                  isRequired: true,
                  onChanged: (value) {
                    controller.calculateUpdateSalePrice(widget.index);
                  },
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                  suffixIcon: CustomAppCard(
                    margin: EdgeInsetsDirectional.only(end: Dimensions.space4),
                    padding: EdgeInsets.all(Dimensions.space8),
                    radius: Dimensions.largeRadius - 2,
                    child: FittedBox(
                        fit: BoxFit.scaleDown,
                        child: Text(
                          SharedPreferenceService.getCurrencySymbol(
                              isFullText: true),
                          style: theme.textTheme.headlineSmall?.copyWith(
                              fontWeight: FontWeight.normal,
                              color: MyColor.getBodyTextColor()
                                  .withValues(alpha: 0.7)),
                        )),
                  ),
                ),
              ),
            ],
          ),
          spaceDown(Dimensions.space16.h),
          Row(
            children: [
              Flexible(
                flex: 1,
                child: CustomDropDownTextField(
                  isRequired: false,
                  selectedValue: controller.formData[widget.index].taxType ??
                      controller.selectedTaxType,
                  onChanged: (value) {
                    controller.selectedTaxType = value;
                    controller.formData[widget.index].taxType = value;
                    controller.calculateUpdateSalePrice(widget.index);

                    controller.update();
                  },
                  labelText: MyStrings.taxType.tr,
                  items: controller.taxTypeData.map((item) {
                    return DropdownMenuItem(
                      value: item["name"],
                      child: Text(item["name"] ?? "",
                          style: theme.textTheme.bodySmall),
                    );
                  }).toList(),
                ),
              ),
              spaceSide(Dimensions.space10),
              Flexible(
                flex: 1,
                child: CustomDropDownTextField(
                  isRequired: false,
                  selectedValue:
                      controller.formData[widget.index].selectedTax ??
                          "Select Tax",
                  onChanged: (value) {
                    final selectedItem = controller.updateTaxes.firstWhere(
                      (item) => item.name == value,
                    );
                    controller.formData[widget.index].taxId = selectedItem.id;
                    controller.formData[widget.index].vatPercentage =
                        selectedItem.percentage;

                    controller.calculateUpdateSalePrice(widget.index);

                    controller.update();
                  },
                  labelText: MyStrings.tax.tr,
                  items: controller.updateTaxes.map((item) {
                    return DropdownMenuItem(
                      value: item.name,
                      child: Text("${item.name} ${item.percentage ?? ""} ",
                          style: theme.textTheme.bodySmall),
                    );
                  }).toList(),
                ),
              ),
            ],
          ),
          spaceDown(Dimensions.space16.h),
          Row(
            children: [
              Flexible(
                flex: 1,
                child: LabelTextField(
                  labelText: MyStrings.purchasePrice.tr,
                  controller: item.purchasePriceController,
                  isRequired: true,
                  onChanged: (value) {
                    controller.formData[widget.index].productPrice = value;
                    controller.calculateUpdateSalePrice(widget.index);
                  },
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                ),
              ),
              spaceSide(Dimensions.space10),
              Flexible(
                flex: 1,
                child: LabelTextField(
                  labelText: MyStrings.profitMargin.tr,
                  isRequired: true,
                  controller: item.profitMarginController,
                  onChanged: (value) {
                    controller.calculateUpdateSalePrice(widget.index);
                  },
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                  suffixIcon: CustomAppCard(
                    margin: EdgeInsetsDirectional.only(end: Dimensions.space4),
                    padding: EdgeInsets.all(Dimensions.space8),
                    radius: Dimensions.largeRadius - 2,
                    child: FittedBox(
                        fit: BoxFit.scaleDown,
                        child: Text(
                          "%",
                          style: theme.textTheme.headlineSmall?.copyWith(
                              fontWeight: FontWeight.normal,
                              color: MyColor.getBodyTextColor()
                                  .withValues(alpha: 0.7)),
                        )),
                  ),
                ),
              ),
            ],
          ),
          spaceDown(Dimensions.space16),
          LabelTextField(
            labelText: MyStrings.salePrice.tr,
            isRequired: true,
            controller: item.salePriceController,
            onChanged: (value) {
              controller.calculateUpdateSalePrice(widget.index,
                  isEditingSalePrice: true);
            },
            contentPadding: EdgeInsetsDirectional.symmetric(
                vertical: Dimensions.space17.h,
                horizontal: Dimensions.space16.w),
            textInputType: TextInputType.number,
            textInputAction: TextInputAction.done,
            radius: Dimensions.largeRadius,
            validator: (value) {
              if (value!.isEmpty) {
                return MyStrings.fieldErrorMsg.tr;
              } else {
                return null;
              }
            },
            suffixIcon: CustomAppCard(
              margin: EdgeInsetsDirectional.only(end: Dimensions.space4),
              padding: EdgeInsets.all(Dimensions.space8),
              radius: Dimensions.largeRadius - 2,
              child: FittedBox(
                  fit: BoxFit.scaleDown,
                  child: Text(
                    SharedPreferenceService.getCurrencySymbol(isFullText: true),
                    style: theme.textTheme.headlineSmall?.copyWith(
                        fontWeight: FontWeight.normal,
                        color:
                            MyColor.getBodyTextColor().withValues(alpha: 0.7)),
                  )),
            ),
          ),
          spaceDown(Dimensions.space16),
          Row(
            children: [
              Flexible(
                flex: 1,
                child: CustomDropDownTextField(
                  isRequired: false,
                  selectedValue:
                      controller.formData[widget.index].discountType ??
                          controller.selectedDiscountType,
                  onChanged: (value) {
                    controller.selectedDiscountType = value;
                    controller.formData[widget.index].discountType = value;

                    controller.calculateUpdateSalePrice(widget.index);
                    controller.update();
                  },
                  labelText: MyStrings.discount.tr,
                  items: controller.discountTypeData.map((item) {
                    return DropdownMenuItem(
                      value: item["name"],
                      child: Text(item["name"] ?? "",
                          style: theme.textTheme.bodySmall),
                    );
                  }).toList(),
                ),
              ),
              spaceSide(Dimensions.space10),
              Flexible(
                flex: 2,
                child: LabelTextField(
                  labelText: "",
                  isRequired: false,
                  controller: item.discountController,
                  onChanged: (value) {
                    controller.calculateUpdateSalePrice(widget.index);
                  },
                  textInputFormatter: [
                    FilteringTextInputFormatter.digitsOnly, // Allow only digits
                    MaxDigitInputFormatter(maxValue: 100)
                  ],
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                  suffixIcon: CustomAppCard(
                    margin: EdgeInsetsDirectional.only(end: Dimensions.space4),
                    padding: EdgeInsets.all(Dimensions.space8),
                    radius: Dimensions.largeRadius - 2,
                    child: FittedBox(
                        fit: BoxFit.scaleDown,
                        child: Text(
                          item.discountType == "Percent"
                              ? "%"
                              : SharedPreferenceService.getCurrencySymbol(),
                          style: theme.textTheme.headlineSmall?.copyWith(
                              fontWeight: FontWeight.normal,
                              color: MyColor.getBodyTextColor()
                                  .withValues(alpha: 0.7)),
                        )),
                  ),
                ),
              ),
            ],
          ),
          spaceDown(Dimensions.space16.h),
          Row(
            children: [
              Flexible(
                flex: 1,
                child: LabelTextField(
                  readOnly: true,
                  labelText: MyStrings.finalPrice.tr,
                  controller: item.finalSalePriceController,
                  isRequired: true,
                  onChanged: (value) {},
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                ),
              ),
              spaceSide(Dimensions.space10),
              Flexible(
                flex: 1,
                child: LabelTextField(
                  labelText: MyStrings.alertQty.tr,
                  isRequired: true,
                  controller: TextEditingController(
                      text: AppConverter.formatNumber(item.alertQuantity ?? "0",
                          precision: 0)),
                  onChanged: (value) {
                    controller.formData[widget.index].alertQuantity = value;
                  },
                  contentPadding: EdgeInsetsDirectional.symmetric(
                      vertical: Dimensions.space17.h,
                      horizontal: Dimensions.space16.w),
                  textInputType: TextInputType.number,
                  textInputAction: TextInputAction.done,
                  radius: Dimensions.largeRadius,
                  validator: (value) {
                    if (value!.isEmpty) {
                      return MyStrings.fieldErrorMsg.tr;
                    } else {
                      return null;
                    }
                  },
                ),
              ),
            ],
          ),
          spaceDown(Dimensions.space16.h),
        ],
      );
    });
  }
}
