import 'dart:async';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/divider/custom_divider.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/categories_shimmer.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/screens/products/widgets/products_details_tile.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/products/products_controller.dart';
import 'package:ovosale/data/repo/products/products_repo.dart';

import '../../../core/app_permission/app_permission.dart';
import '../../../data/controller/cart/global_controller.dart';

class ProductScreen extends StatefulWidget {
  const ProductScreen({super.key});

  @override
  State<ProductScreen> createState() => _ProductScreenState();
}

class _ProductScreenState extends State<ProductScreen> {
  final ScrollController scrollController = ScrollController();

  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<ProductsController>().hasNext()) {
        Get.find<ProductsController>().getProductData(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;
  @override
  void initState() {
    Get.put(ProductsRepo());
    final controller = Get.put(ProductsController(productsRepo: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.getProductData();
      scrollController.addListener(scrollListener);
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<ProductsController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.products.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            Container(
              padding: EdgeInsetsDirectional.symmetric(
                  horizontal: Dimensions.space16.w,
                  vertical: Dimensions.space12.h),
              color: MyColor.white,
              child: LabelTextField(
                labelText: "",
                hideLabel: true,
                hintText: MyStrings.search.tr,
                controller: controller.productSearchController,
                hintTextColor: MyColor.getBodyTextColor(),
                onChanged: (value) {
                  if (_debounceTimer?.isActive ?? false) {
                    _debounceTimer?.cancel();
                  }

                  _debounceTimer = Timer(const Duration(milliseconds: 500), () {
                    controller.getProductData();
                  });
                },
                textInputType: TextInputType.text,
                textInputAction: TextInputAction.done,
                radius: Dimensions.largeRadius,
                validator: (value) {
                  if (value!.isEmpty) {
                    return MyStrings.fieldErrorMsg.tr;
                  } else {
                    return null;
                  }
                },
                prefixIcon: Icon(
                  CupertinoIcons.search,
                  size: Dimensions.textFieldIconSize,
                ),
              ),
            ),
            spaceDown(Dimensions.space10),
            Expanded(
              child: CustomAppCard(
                radius: 0,
                showBorder: false,
                child: controller.isLoading
                    ? CategoriesShimmer()
                    : controller.filteredProducts.isEmpty
                        ? NoDataWidget()
                        : ListView.separated(
                            physics: BouncingScrollPhysics(),
                            controller: scrollController,
                            separatorBuilder: (context, index) =>
                                CustomDivider(space: Dimensions.space16.h),
                            shrinkWrap: true,
                            itemCount: controller.hasNext()
                                ? controller.filteredProducts.length + 1
                                : controller.filteredProducts.length,
                            itemBuilder: (context, i) {
                              if (i == controller.filteredProducts.length) {
                                return controller.hasNext()
                                    ? SizedBox(
                                        height: Dimensions.space40,
                                        width:
                                            MediaQuery.of(context).size.width,
                                        child: const CustomLoader(
                                          isPagination: true,
                                        ),
                                      )
                                    : const SizedBox();
                              }
                              final product = controller.filteredProducts[i];
                              return GestureDetector(
                                onTap: () {
                                  Get.toNamed(RouteHelper.productDetailsScreen,
                                      arguments: [product.id]);
                                },
                                child: ProductsDetailsListTile(
                                  index: i,
                                  productData: controller.filteredProducts,
                                  productDetails: controller.productDetails,
                                ),
                              );
                            },
                          ),
              ),
            ),
          ],
        ),
        floatingActionButton: CustomAppCard(
            onPressed: () {
              if (Get.find<GlobalController>()
                  .checkPermission(featureName: AppPermissions.addProduct)) {
                Get.toNamed(RouteHelper.newProductScreen)?.then((v) {
                  controller.getProductData();
                });
              }
            },
            height: Dimensions.space64.h,
            width: Dimensions.space64.w,
            padding: EdgeInsetsDirectional.all(Dimensions.space20),
            backgroundColor: MyColor.getPrimaryColor(),
            showBorder: false,
            child: MyAssetImageWidget(
              color: theme.cardColor,
              assetPath: MyIcons.add,
              isSvg: true,
            )),
      ),
    );
  }
}
