import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:bluetooth_print_plus/bluetooth_print_plus.dart';
import 'package:ovosale/core/utils/my_strings.dart';
import 'package:ovosale/core/utils/util.dart';
import 'package:ovosale/data/controller/payment/payment_controller.dart';

class SelectDevicePopUp extends StatefulWidget {
  const SelectDevicePopUp({super.key, this.isPrintable = true});
  final bool isPrintable;

  @override
  State<SelectDevicePopUp> createState() => _SelectDevicePopUpState();
}

class _SelectDevicePopUpState extends State<SelectDevicePopUp> {
  bool isScanning = true;
  List<BluetoothDevice> devices = [];

  @override
  void initState() {
    super.initState();
    _startScanning();
  }

  void _startScanning() async {
    final controller = Get.find<PaymentController>();

    try {
      setState(() => isScanning = true);

      // Start scanning using the new service
      await controller.startScan();

      // Listen to scan results
      controller.scanResults.listen((deviceList) {
        if (mounted) {
          setState(() {
            devices = deviceList;
          });
        }
      });

      // Stop scanning indicator after timeout
      Future.delayed(const Duration(seconds: 10), () {
        if (mounted) {
          setState(() => isScanning = false);
        }
      });
    } catch (e) {
      printX('Scanning error: $e');
      if (mounted) {
        setState(() => isScanning = false);
      }
    }
  }

  void _refreshScan() {
    setState(() {
      devices.clear();
      isScanning = true;
    });
    _startScanning();
  }

  @override
  Widget build(BuildContext context) {
    return GetBuilder<PaymentController>(builder: (controller) {
      return Container(
        padding: const EdgeInsets.all(16),
        decoration: BoxDecoration(
          color: Colors.white,
          borderRadius: BorderRadius.circular(12),
        ),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            // Header
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Text(
                  MyStrings.selectDevice.tr,
                  style: const TextStyle(
                    fontSize: 18,
                    fontWeight: FontWeight.bold,
                  ),
                ),
                IconButton(
                  onPressed: _refreshScan,
                  icon: Icon(
                    isScanning ? Icons.stop : Icons.refresh,
                    color: isScanning ? Colors.red : Colors.blue,
                  ),
                ),
              ],
            ),

            const SizedBox(height: 16),

            // Scanning indicator
            if (isScanning)
              const Padding(
                padding: EdgeInsets.all(16.0),
                child: Column(
                  children: [
                    CircularProgressIndicator(),
                    SizedBox(height: 12),
                    Text('Scanning for devices...'),
                  ],
                ),
              ),

            // Device list
            Container(
              constraints: const BoxConstraints(
                maxHeight: 300, // Limit height to prevent overflow
              ),
              child: devices.isEmpty && !isScanning
                  ? _buildEmptyState()
                  : _buildDeviceList(controller),
            ),

            const SizedBox(height: 16),

            // Close button
            SizedBox(
              width: double.infinity,
              child: ElevatedButton(
                onPressed: () => Get.back(),
                style: ElevatedButton.styleFrom(
                  backgroundColor: Colors.grey[300],
                  foregroundColor: Colors.black87,
                ),
                child: const Text('Close'),
              ),
            ),
          ],
        ),
      );
    });
  }

  Widget _buildEmptyState() {
    return const Center(
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Icon(
            Icons.bluetooth_searching,
            size: 48,
            color: Colors.grey,
          ),
          SizedBox(height: 12),
          Text('No printer devices found'),
          SizedBox(height: 8),
          Text(
            'Make sure your printer is turned on and in pairing mode',
            style: TextStyle(
              fontSize: 12,
              color: Colors.grey,
            ),
            textAlign: TextAlign.center,
          ),
        ],
      ),
    );
  }

  Widget _buildDeviceList(PaymentController controller) {
    return StreamBuilder<List<BluetoothDevice>>(
      stream: controller.scanResults,
      initialData: devices,
      builder: (context, snapshot) {
        final deviceList = snapshot.data ?? devices;

        if (deviceList.isEmpty && !isScanning) {
          return _buildEmptyState();
        }

        return ListView.builder(
          shrinkWrap: true,
          itemCount: deviceList.length,
          itemBuilder: (context, index) {
            final device = deviceList[index];
            final isSelected = controller.selectedDevice != null &&
                controller.selectedDevice!.address == device.address;

            return Card(
              margin: const EdgeInsets.symmetric(vertical: 4),
              child: ListTile(
                leading: Icon(
                  Icons.print,
                  color: isSelected ? Colors.green : Colors.blue,
                ),
                title: Text(
                  device.name.isNotEmpty == true
                      ? device.name
                      : 'Unknown device',
                  style: TextStyle(
                    fontWeight:
                        isSelected ? FontWeight.bold : FontWeight.normal,
                  ),
                ),
                subtitle: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      device.address,
                      style: const TextStyle(fontSize: 12),
                    ),
                    Text(
                      'Type: ${device.type}',
                      style: const TextStyle(
                        fontSize: 11,
                        color: Colors.grey,
                      ),
                    ),
                  ],
                ),
                trailing: isSelected
                    ? const Icon(Icons.check_circle, color: Colors.green)
                    : const Icon(Icons.radio_button_unchecked,
                        color: Colors.grey),
                onTap: () {
                  try {
                    controller.selectDevice(device);
                    printX('Selected device: ${device.toJson()}');

                    // Connect to device
                    controller.connectDevice(isPrintable: widget.isPrintable);
                  } catch (e) {
                    printX('Device selection error: $e');
                    Get.snackbar(
                      'Error',
                      'Failed to select device: $e',
                      backgroundColor: Colors.red[100],
                      colorText: Colors.red[800],
                    );
                  }
                },
              ),
            );
          },
        );
      },
    );
  }

  @override
  void dispose() {
    // Stop scanning when widget is disposed
    final controller = Get.find<PaymentController>();
    controller.stopScan();
    super.dispose();
  }
}
