import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/alert-dialog/custom_alert_dialog.dart';
import 'package:ovosale/app/components/buttons/custom_elevated_button.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/shimmer/payment_shimmer.dart';
import 'package:ovosale/app/screens/payment/widgets/payment_option_section.dart';
import 'package:ovosale/app/screens/payment/widgets/select_device_pop_up.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/payment/payment_controller.dart';
import 'package:ovosale/data/repo/payment/payment_repo.dart';
import 'package:ovosale/data/services/shared_pref_service.dart';

import '../../components/text-field/label_text_field.dart';

class PaymentScreen extends StatefulWidget {
  const PaymentScreen({super.key});

  @override
  State<PaymentScreen> createState() => _PaymentScreenState();
}

class _PaymentScreenState extends State<PaymentScreen> {
  @override
  void initState() {
    Get.put(PaymentRepo());
    final controller = Get.put(PaymentController(paymentRepo: Get.find()));

    super.initState();
    controller.clearOldData();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) async {
      await controller.getPaymentMethod();
    });
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<PaymentController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.payment.tr,
        actionButton: [
          IconButton(
              onPressed: () {
                CustomAlertDialog(
                    child: SelectDevicePopUp(
                  isPrintable: false,
                )).customAlertDialog(context);
              },
              icon: Icon(
                controller.connected
                    ? Icons.print_rounded
                    : Icons.print_disabled_rounded,
                color: MyColor.getBodyTextColor(),
              )),
          spaceSide(Dimensions.space10)
        ],
        body: SingleChildScrollView(
          child: Column(
            children: [
              CustomAppCard(
                radius: Dimensions.space12,
                width: double.infinity,
                padding: EdgeInsetsDirectional.all(Dimensions.space20),
                backgroundColor: theme.cardColor,
                borderColor: theme.primaryColor,
                child: Row(
                  children: [
                    Expanded(
                        child: Text(MyStrings.totalPaybleAmount.tr,
                            style: theme.textTheme.headlineSmall)),
                    CustomAppCard(
                        radius: Dimensions.space8,
                        backgroundColor: theme.primaryColor,
                        padding: EdgeInsetsDirectional.symmetric(
                            horizontal: Dimensions.space10,
                            vertical: Dimensions.space2.h),
                        showBorder: false,
                        child: Text(
                            "${SharedPreferenceService.getCurrencySymbol()}${controller.billingController.getTotal()}",
                            style: theme.textTheme.headlineSmall?.copyWith(
                              color: MyColor.white,
                            )))
                  ],
                ),
              ),
              SizedBox(height: Dimensions.space18.h),
              CustomAppCard(
                  width: double.infinity,
                  showBorder: false,
                  radius: Dimensions.space12,
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        MyStrings.customer.tr,
                        style: theme.textTheme.labelSmall?.copyWith(
                            fontWeight: FontWeight.w600, color: MyColor.black),
                      ),
                      SizedBox(height: Dimensions.space8.h),
                      CustomAppCard(
                        onPressed: () {
                          Get.toNamed(RouteHelper.customerDataScreen,
                                  arguments: true)
                              ?.then((value) {
                            controller.customerData = value[0];
                            controller.update();
                          });
                        },
                        radius: Dimensions.space12,
                        width: double.infinity,
                        borderRadius: BorderRadius.circular(8),
                        padding: EdgeInsetsDirectional.symmetric(
                            horizontal: Dimensions.space12, vertical: 10),
                        backgroundColor: MyColor.getBorderColor(),
                        borderColor: MyColor.getBorderColor(),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            Expanded(
                                child: Text(
                                    controller.customerData.id != null
                                        ? controller.customerData.name ?? ""
                                        : MyStrings.walkinCustomer.tr,
                                    style: theme.textTheme.titleSmall)),
                            Icon(
                              Icons.arrow_forward_ios_rounded,
                              size: 16,
                              color: MyColor.lightBodyText,
                            )
                          ],
                        ),
                      )
                    ],
                  )),
              controller.isLoading ? PaymentShimmer() : PaymentOptionSection(),
              SizedBox(height: Dimensions.space18.h),
              CustomAppCard(
                  width: double.infinity,
                  showBorder: false,
                  radius: Dimensions.space12,
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        MyStrings.optional.tr,
                        style: theme.textTheme.labelSmall?.copyWith(
                            fontWeight: FontWeight.w600, color: MyColor.black),
                      ),
                      SizedBox(height: Dimensions.space8.h),
                      LabelTextField(
                        controller: controller.noteController,
                        labelText: "",
                        hideLabel: true,
                        maxLines: 3,
                        hintText: MyStrings.notes.tr,
                        fillColor: MyColor.lightBodyText.withValues(alpha: .1),
                        hintTextColor: MyColor.getBodyTextColor(),
                        onChanged: (value) {},
                        textInputType: TextInputType.text,
                        textInputAction: TextInputAction.done,
                        radius: Dimensions.largeRadius,
                        validator: (value) {
                          if (value!.isEmpty) {
                            return MyStrings.fieldErrorMsg.tr;
                          } else {
                            return null;
                          }
                        },
                      ),
                    ],
                  )),
              spaceDown(Dimensions.space30),
              CustomElevatedBtn(
                isLoading: controller.isConfirmLoading,
                text: MyStrings.confirmSale.tr,
                onTap: () {
                  controller.onConfirmSale();
                },
                height: Dimensions.space56.h,
                radius: Dimensions.largeRadius,
              ),
              spaceDown(Dimensions.space14.h),
              CustomElevatedBtn(
                isLoading: controller.isSaleAndPrintLoading,
                bgColor: MyColor.getBorderColor(),
                text: MyStrings.confirmSalesAndPrint.tr,
                textColor: MyColor.black,
                onTap: () async {
                  if (await controller.checkBluetoothStatus() == true) {
                    if (controller.connected) {
                      controller.onConfirmSale(isSaleAndPrint: true);
                    } else {
                      CustomAlertDialog(child: SelectDevicePopUp())
                          .customAlertDialog(context);
                    }
                  }
                },
                height: Dimensions.space56.h,
                radius: Dimensions.largeRadius,
              ),
              spaceDown(Dimensions.space14.h),
              CustomElevatedBtn(
                isLoading: controller.isPdfDownloadLoading,
                bgColor: MyColor.getBorderColor(),
                text: MyStrings.confirmSalesAndSavePdf.tr,
                textColor: MyColor.black,
                onTap: () {
                  controller.onConfirmSale(isSaleAndSavePdf: true);
                },
                height: Dimensions.space56.h,
                radius: Dimensions.largeRadius,
              )
            ],
          ),
        ),
      ),
    );
  }
}
