import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/shimmer/new_sale_shimmer.dart';
import 'package:ovosale/app/screens/new_sale/widgets/all_products_section.dart';
import 'package:ovosale/app/screens/new_sale/widgets/app_bar_section.dart';
import 'package:ovosale/app/screens/new_sale/widgets/category_section.dart';
import 'package:ovosale/app/screens/new_sale/widgets/warehouse_section.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/cart/global_controller.dart';
import 'package:ovosale/data/controller/new_sale/new_sale_controller.dart';
import 'package:ovosale/data/repo/new_sale/new_sale_repo.dart';

import '../../components/cart_count_widget.dart';

class NewSaleScreen extends StatefulWidget {
  const NewSaleScreen({super.key});

  @override
  State<NewSaleScreen> createState() => _NewSaleScreenState();
}

class _NewSaleScreenState extends State<NewSaleScreen> {
  final ScrollController historyScrollController = ScrollController();
  void fetchData() {
    Get.find<NewSaleController>().loadProductData(isPaginate: true);
  }

  void scrollListener() {
    if (historyScrollController.position.pixels ==
        historyScrollController.position.maxScrollExtent) {
      if (Get.find<NewSaleController>().hasNext()) {
        fetchData();
      }
    }
  }

  @override
  void initState() {
    Get.put(NewSaleRepo());
    final controller = Get.put(NewSaleController(newSaleRepo: Get.find()));

    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.loadData();

      if (mounted) {
        // Add scroll listeners
        historyScrollController.addListener(() => scrollListener());
      }
    });
  }

  @override
  void dispose() {
    historyScrollController.dispose();
    Get.find<GlobalController>().page = 1;
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<NewSaleController>(
      builder: (controller) => MyCustomScaffold(
        padding: EdgeInsetsDirectional.zero,
        pageTitle: MyStrings.newSale.tr,
        actionButton: [
          GestureDetector(
            onTap: () {
              showModalBottomSheet(
                  shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.only(
                          topLeft: Radius.circular(15),
                          topRight: Radius.circular(15))),
                  isScrollControlled: true,
                  useSafeArea: true,
                  context: context,
                  builder: (context) {
                    return WarehouseSection();
                  });
            },
            child: Container(
                padding: EdgeInsets.symmetric(horizontal: 8, vertical: 4),
                decoration: BoxDecoration(
                  borderRadius: BorderRadius.circular(4),
                  border: Border.all(
                    color: MyColor.getBorderColor(),
                  ),
                ),
                child: Row(
                  children: [
                    Icon(
                      Icons.place_rounded,
                      color: theme.primaryColor,
                      size: 16,
                    ),
                    SizedBox(width: 4.w),
                    Text(
                      Get.find<GlobalController>().selectedWareHouse.name ?? "",
                      style: theme.textTheme.titleSmall?.copyWith(fontSize: 12),
                    )
                  ],
                )),
          ),
          Padding(
              padding: const EdgeInsets.only(right: 10),
              child: CartCountWidget()),
        ],
        body: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            AppBarSection(),
            controller.isLoading
                ? NewSaleShimmer()
                : Expanded(
                    child: SingleChildScrollView(
                      controller: historyScrollController,
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Padding(
                            padding: EdgeInsets.symmetric(
                                horizontal: Dimensions.space16.w,
                                vertical: Dimensions.space12.h),
                            child: Text(MyStrings.category.tr),
                          ),
                          CategorySection(),
                          Padding(
                            padding: EdgeInsets.symmetric(
                                horizontal: Dimensions.space16.w,
                                vertical: Dimensions.space12.h),
                            child: Text(MyStrings.products.tr),
                          ),
                          AllProductsSection(
                              historyScrollController: historyScrollController),
                          controller.hasNext()
                              ? const CustomLoader(isPagination: true)
                              : const SizedBox(),
                        ],
                      ),
                    ),
                  )
          ],
        ),
        bottomNavigationBar: GetBuilder<GlobalController>(
          builder: (controller) => CustomAppCard(
              margin: EdgeInsets.symmetric(
                  vertical: Dimensions.space25.h,
                  horizontal: Dimensions.space16.w),
              onPressed: () {
                Get.toNamed(RouteHelper.checkoutScreen)?.then((value) {
                  controller.getCartCount();
                });
              },
              height: Dimensions.space64.h,
              width: double.infinity,
              backgroundColor: MyColor.getPrimaryColor().withValues(alpha: .45),
              borderColor: theme.primaryColor,
              child: Row(
                children: [
                  Expanded(
                      child: Text(MyStrings.viewCartAndCheckout,
                          style: theme.textTheme.headlineSmall)),
                  CustomAppCard(
                      alignment: Alignment.center,
                      radius: Dimensions.space8,
                      backgroundColor: theme.primaryColor,
                      padding: EdgeInsetsDirectional.symmetric(
                          horizontal: Dimensions.space10,
                          vertical: Dimensions.space2.h),
                      showBorder: false,
                      child: Text(controller.cartCount,
                          textAlign: TextAlign.center,
                          style: theme.textTheme.headlineSmall?.copyWith(
                            color: MyColor.white,
                          )))
                ],
              )),
        ),
      ),
    );
  }
}
