import 'dart:ui';

import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/buttons/custom_elevated_button.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/text-field/custom_drop_down_text_field.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/components/text/label_text.dart';
import 'package:ovosale/app/screens/expense/widgets/pick_image_section.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/expense/new_expense_controller.dart';
import 'package:ovosale/data/repo/expense/new_expense_repo.dart';

class NewExpenseScreen extends StatefulWidget {
  const NewExpenseScreen({super.key});

  @override
  State<NewExpenseScreen> createState() => _NewExpenseScreenState();
}

class _NewExpenseScreenState extends State<NewExpenseScreen> {
  @override
  void initState() {
    Get.put(NewExpenseRepo());
    final controller =
        Get.put(NewExpenseController(newExpenseRepo: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.getExpenseCategoryData();
    });
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<NewExpenseController>(
      builder: (controller) => MyCustomScaffold(
        padding: EdgeInsetsDirectional.zero,
        pageTitle: MyStrings.newExpense.tr,
        body: controller.loading
            ? CustomLoader()
            : SingleChildScrollView(
                child: CustomAppCard(
                    radius: 0,
                    margin: EdgeInsetsDirectional.symmetric(
                        vertical: Dimensions.space10.h),
                    showBorder: false,
                    child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          LabelTextField(
                            isRequired: true,
                            onChanged: () {},
                            readOnly: true,
                            controller: controller.expenseDateController,
                            labelText: MyStrings.expenseDate.tr,
                            textInputType: TextInputType.phone,
                            onTap: () {
                              controller
                                  .changeEndSelectedDateOnlyValue(context);
                            },
                            textInputAction: TextInputAction.next,
                            radius: Dimensions.largeRadius,
                            validator: (value) {
                              if (value!.isEmpty) {
                                return MyStrings.fieldErrorMsg.tr;
                              } else {
                                return null;
                              }
                            },
                          ),
                          spaceDown(Dimensions.space16.h),
                          CustomDropDownTextField(
                            selectedValue: controller.selectedValue,
                            onChanged: (value) {
                              final selectedAttribute =
                                  controller.expenseDataList.firstWhere(
                                (attribute) => attribute.name == value,
                              );
                              controller.selectedValue = value;
                              controller.expensePurposeId =
                                  selectedAttribute.id ?? "";

                              controller.update();
                            },
                            labelText: MyStrings.expensePurpose.tr,
                            items: controller.expenseDataList.map((attribute) {
                              return DropdownMenuItem<String>(
                                value: attribute.name,
                                child: Text(attribute.name ?? 'Unknown',
                                    style: theme.textTheme.bodySmall),
                              );
                            }).toList(),
                          ),
                          spaceDown(Dimensions.space16.h),
                          LabelTextField(
                            controller: controller.referenceController,
                            labelText: MyStrings.referenceNo.tr,
                            onChanged: (value) {},
                            contentPadding: EdgeInsetsDirectional.symmetric(
                                vertical: Dimensions.space17.h,
                                horizontal: Dimensions.space16.w),
                            textInputType: TextInputType.number,
                            textInputAction: TextInputAction.done,
                            radius: Dimensions.largeRadius,
                            validator: (value) {
                              if (value!.isEmpty) {
                                return MyStrings.fieldErrorMsg.tr;
                              } else {
                                return null;
                              }
                            },
                          ),
                          spaceDown(Dimensions.space16.h),
                          LabelTextField(
                            controller: controller.commentController,
                            labelText: MyStrings.comment.tr,
                            onChanged: (value) {},
                            contentPadding: EdgeInsetsDirectional.symmetric(
                                vertical: Dimensions.space17.h,
                                horizontal: Dimensions.space16.w),
                            textInputType: TextInputType.text,
                            textInputAction: TextInputAction.done,
                            radius: Dimensions.largeRadius,
                            validator: (value) {
                              if (value!.isEmpty) {
                                return MyStrings.fieldErrorMsg.tr;
                              } else {
                                return null;
                              }
                            },
                          ),
                          spaceDown(Dimensions.space16.h),
                          LabelTextField(
                            controller: controller.amountController,
                            labelText: MyStrings.amount.tr,
                            isRequired: true,
                            onChanged: (value) {},
                            contentPadding: EdgeInsetsDirectional.symmetric(
                                vertical: Dimensions.space17.h,
                                horizontal: Dimensions.space16.w),
                            textInputType: TextInputType.number,
                            textInputAction: TextInputAction.done,
                            radius: Dimensions.largeRadius,
                            validator: (value) {
                              if (value!.isEmpty) {
                                return MyStrings.fieldErrorMsg.tr;
                              } else {
                                return null;
                              }
                            },
                          ),
                          spaceDown(Dimensions.space16.h),
                          Column(
                            children: List.generate(
                                controller.paymentMethodList.length, (index) {
                              return CustomAppCard(
                                onPressed: () {
                                  controller.changeSelectedPaymentMethod(index);
                                },
                                margin: EdgeInsets.only(top: 12.h),
                                radius: Dimensions.space12,
                                width: double.infinity,
                                padding: EdgeInsetsDirectional.all(
                                    Dimensions.space4),
                                backgroundColor: theme.cardColor,
                                borderColor:
                                    controller.paymentMethodCurrentIndex ==
                                            index
                                        ? theme.primaryColor
                                        : MyColor.getBorderColor(),
                                child: Row(
                                  children: [
                                    CustomAppCard(
                                        radius: Dimensions.space8,
                                        backgroundColor:
                                            MyColor.getSectionBackgroundColor(),
                                        padding: EdgeInsetsDirectional.all(
                                            Dimensions.space12.h),
                                        showBorder: false,
                                        child: MyAssetImageWidget(
                                          assetPath: controller
                                                      .paymentMethodList[index]
                                                      .name ==
                                                  "Card"
                                              ? MyIcons.cardPayment
                                              : MyIcons.cashPayment,
                                          isSvg: true,
                                          height: Dimensions.space24.h,
                                          width: Dimensions.space24.w,
                                        )),
                                    spaceSide(Dimensions.space8),
                                    Expanded(
                                        child: Text(
                                            controller.paymentMethodList[index]
                                                    .name ??
                                                "",
                                            style: theme.textTheme.bodyLarge
                                                ?.copyWith(
                                                    color: MyColor.black))),
                                  ],
                                ),
                              );
                            }),
                          ),
                          spaceDown(Dimensions.space15),
                          if (controller
                              .paymentAccountsDataList.isNotEmpty) ...[
                            spaceDown(Dimensions.space12),
                            CustomDropDownTextField(
                              selectedValue:
                                  controller.paymentAccountsDataList.isNotEmpty
                                      ? controller.paymentAccountsDataList.first
                                      : null,
                              onChanged: (value) {
                                controller.setSelectedPaymentAccount(value);
                              },
                              labelText: MyStrings.paymentAccount.tr,
                              items: controller.paymentAccountsDataList
                                  .map((item) {
                                return DropdownMenuItem(
                                  value: item,
                                  child: Text(
                                    item.accountName ?? "",
                                    style: theme.textTheme.bodySmall,
                                  ),
                                );
                              }).toList(),
                            ),
                          ],
                          spaceDown(Dimensions.space16.h),
                          LabelText(
                            text: MyStrings.attachment.tr,
                            textStyle: theme.textTheme.labelMedium?.copyWith(
                                fontSize: Dimensions.space15.sp,
                                color: MyColor.black),
                          ),
                          spaceDown(Dimensions.space10.h),
                          CustomPaint(
                            painter: DottedBorderPainter(),
                            child: CustomAppCard(
                              showBorder: false,
                              width: double.infinity,
                              child: Stack(
                                children: [
                                  Align(
                                    alignment: Alignment.center,
                                    child: GestureDetector(
                                      onTap: () {
                                        controller.pickFile();
                                      },
                                      child: controller.selectedFile == null
                                          ? Column(
                                              children: [
                                                MyAssetImageWidget(
                                                    assetPath: MyIcons.addImage,
                                                    isSvg: true,
                                                    height:
                                                        Dimensions.space56.h,
                                                    width: Dimensions.space56.w,
                                                    color: theme.primaryColor),
                                                Text(
                                                  MyStrings.tapToAddImage.tr,
                                                  style: theme
                                                      .textTheme.bodyLarge
                                                      ?.copyWith(
                                                    fontSize:
                                                        Dimensions.space16.sp,
                                                  ),
                                                )
                                              ],
                                            )
                                          : PickImageSection(),
                                    ),
                                  ),
                                  if (controller.selectedFile != null)
                                    Align(
                                      alignment: Alignment
                                          .topRight, // Position the button in the top-right
                                      child: CustomAppCard(
                                        onPressed: () {
                                          controller.removeSelectedFile();
                                        },
                                        showBorder: false,
                                        padding: EdgeInsetsDirectional.all(
                                            Dimensions.space10),
                                        radius: Dimensions.space8,
                                        backgroundColor: MyColor.getErrorColor()
                                            .withValues(alpha: .2),
                                        child: MyAssetImageWidget(
                                          assetPath: MyIcons.delete,
                                          isSvg: true,
                                          height: Dimensions.space16.h,
                                          width: Dimensions.space16.w,
                                        ),
                                      ),
                                    ),
                                ],
                              ),
                            ),
                          ),
                          spaceDown(Dimensions.space25.h),
                          CustomElevatedBtn(
                            isLoading: controller.isSubmitting,
                            text: MyStrings.save.tr,
                            onTap: () {
                              controller.createExpense();
                            },
                            height: Dimensions.space56.h,
                            radius: Dimensions.largeRadius,
                          ),
                        ])),
              ),
      ),
    );
  }
}

class DottedBorderPainter extends CustomPainter {
  @override
  void paint(Canvas canvas, Size size) {
    final paint = Paint()
      ..color = MyColor.getPrimaryColor()
      ..strokeWidth = 2
      ..style = PaintingStyle.stroke;

    final double dashWidth = 3; // Width of each dash
    final double dashSpace = 2; // Space between each dash
    final double cornerRadius = 10; // Radius for rounded corners

    final path = Path()
      ..addRRect(RRect.fromRectAndRadius(
          Rect.fromLTWH(0, 0, size.width, size.height),
          Radius.circular(cornerRadius)));

    final PathMetrics pathMetrics = path.computeMetrics();

    for (PathMetric pathMetric in pathMetrics) {
      double distance = 0;
      while (distance < pathMetric.length) {
        final Tangent? tangent = pathMetric.getTangentForOffset(distance);
        if (tangent != null) {
          final Offset start = tangent.position;
          final Offset end = tangent.position +
              Offset(
                  dashWidth * tangent.vector.dx, dashWidth * tangent.vector.dy);
          canvas.drawLine(start, end, paint);
        }
        distance += dashWidth + dashSpace;
      }
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}
