import 'dart:async';

import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/divider/custom_divider.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/customer_database_Shimmer.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/screens/customer/widgets/customers_list_tile.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/customer/customer_controller.dart';
import 'package:ovosale/data/repo/customer/customer_repo.dart';

import '../../../core/app_permission/app_permission.dart';
import '../../../data/controller/cart/global_controller.dart';

class CustomerScreen extends StatefulWidget {
  final bool isFromSale;

  const CustomerScreen({super.key, required this.isFromSale});

  @override
  State<CustomerScreen> createState() => _CustomerScreenState();
}

class _CustomerScreenState extends State<CustomerScreen> {
  final ScrollController scrollController = ScrollController();
  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<CustomerController>().hasNext()) {
        Get.find<CustomerController>().getCustomerData(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;

  @override
  void initState() {
    Get.put(CustomerRepo());
    final controller = Get.put(CustomerController(customerRepo: Get.find()));
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      scrollController.addListener(scrollListener);
      controller.getCustomerData();
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<CustomerController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.customer.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            Container(
              padding: EdgeInsetsDirectional.symmetric(
                  horizontal: Dimensions.space16.w,
                  vertical: Dimensions.space12.h),
              color: MyColor.white,
              child: LabelTextField(
                labelText: "",
                hideLabel: true,
                hintText: MyStrings.search.tr,
                controller: controller.customerSearchController,
                hintTextColor: MyColor.getBodyTextColor(),
                onChanged: (value) {
                  if (_debounceTimer?.isActive ?? false) {
                    _debounceTimer?.cancel();
                  }

                  _debounceTimer = Timer(const Duration(milliseconds: 500), () {
                    controller.getCustomerData();
                  });
                },
                textInputType: TextInputType.text,
                textInputAction: TextInputAction.search,
                radius: Dimensions.largeRadius,
                validator: (value) {
                  if (value!.isEmpty) {
                    return MyStrings.fieldErrorMsg.tr;
                  } else {
                    return null;
                  }
                },
                prefixIcon: Padding(
                  padding: const EdgeInsets.all(10),
                  child: MyAssetImageWidget(
                    assetPath: MyImages.search,
                    isSvg: true,
                    width: 16,
                    height: 16,
                    color: MyColor.lightBodyText,
                  ),
                ),
              ),
            ),
            spaceDown(Dimensions.space10),
            controller.isLoading
                ? CustomerDatabaseShimmer()
                : controller.filteredCustomersDataList.isEmpty &&
                        !controller.isLoading
                    ? Center(
                        child: SizedBox(
                          height: MediaQuery.of(context).size.height * 0.5,
                          child: NoDataWidget(),
                        ),
                      )
                    : Expanded(
                        child: CustomAppCard(
                            radius: 0,
                            showBorder: false,
                            child: ListView.separated(
                                physics: BouncingScrollPhysics(),
                                controller: scrollController,
                                separatorBuilder: (context, index) =>
                                    CustomDivider(space: Dimensions.space16.h),
                                itemCount: controller
                                        .filteredCustomersDataList.length +
                                    1,
                                itemBuilder: (context, i) {
                                  if (i ==
                                      controller
                                          .filteredCustomersDataList.length) {
                                    return controller.hasNext()
                                        ? SizedBox(
                                            height: Dimensions.space40,
                                            width: MediaQuery.of(context)
                                                .size
                                                .width,
                                            child: const CustomLoader(
                                                isPagination: true),
                                          )
                                        : const SizedBox();
                                  }
                                  return GestureDetector(
                                    onTap: () {
                                      if (widget.isFromSale) {
                                        Get.back(result: [
                                          controller
                                              .filteredCustomersDataList[i]
                                        ]);
                                      }
                                    },
                                    child: CustomersListTile(
                                      isShowTrilling:
                                          widget.isFromSale ? false : true,
                                      index: i,
                                    ),
                                  );
                                })),
                      )
          ],
        ),
        floatingActionButton:
            // widget.isFromSale
            //     ? SizedBox()
            //     :
            CustomAppCard(
                onPressed: () {
                  if (Get.find<GlobalController>().checkPermission(
                      featureName: AppPermissions.addCustomer)) {
                    Get.toNamed(RouteHelper.addNewCustomerScreen)?.then((v) {
                      controller.getCustomerData();
                    });
                  }
                },
                height: Dimensions.space64.h,
                width: Dimensions.space64.w,
                padding: EdgeInsetsDirectional.all(Dimensions.space20),
                backgroundColor: MyColor.getPrimaryColor(),
                showBorder: false,
                child: MyAssetImageWidget(
                  color: theme.cardColor,
                  assetPath: MyIcons.add,
                  isSvg: true,
                )),
      ),
    );
  }
}
