import 'dart:async';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/divider/custom_divider.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/variant_shimmer.dart';
import 'package:ovosale/app/components/snack_bar/show_custom_snackbar.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/screens/coupon/widgets/coupon_list_tile.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/coupon/coupon_controller.dart';
import 'package:ovosale/data/model/coupon/coupon_response_model.dart';
import 'package:ovosale/data/repo/coupon/coupon_repo.dart';

import '../../../core/app_permission/app_permission.dart';
import '../../../data/controller/cart/global_controller.dart';

class CouponScreen extends StatefulWidget {
  final bool isFromSale;

  const CouponScreen({super.key, required this.isFromSale});

  @override
  State<CouponScreen> createState() => _CouponScreenState();
}

class _CouponScreenState extends State<CouponScreen> {
  final ScrollController scrollController = ScrollController();
  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<CouponController>().hasNext()) {
        widget.isFromSale
            ? Get.find<CouponController>().getSaleCouponData()
            : Get.find<CouponController>().getCouponData(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;
  @override
  void initState() {
    Get.put(CouponRepo());
    final controller = Get.put(CouponController(
        couponRepo: Get.find(), isFromSale: widget.isFromSale));
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      scrollController.addListener(scrollListener);
      widget.isFromSale
          ? controller.getSaleCouponData()
          : controller.getCouponData();
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<CouponController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.coupon.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            !widget.isFromSale
                ? Container(
                    padding: EdgeInsetsDirectional.symmetric(
                        horizontal: Dimensions.space16.w,
                        vertical: Dimensions.space12.h),
                    color: MyColor.white,
                    child: LabelTextField(
                      labelText: "",
                      hideLabel: true,
                      hintText: MyStrings.search.tr,
                      controller: controller.searchCouponController,
                      hintTextColor: MyColor.getBodyTextColor(),
                      onChanged: (value) {
                        if (_debounceTimer?.isActive ?? false) {
                          _debounceTimer?.cancel();
                        }

                        _debounceTimer =
                            Timer(const Duration(milliseconds: 500), () {
                          controller.getCouponData();
                        });
                      },
                      textInputType: TextInputType.text,
                      textInputAction: TextInputAction.done,
                      radius: Dimensions.largeRadius,
                      validator: (value) {
                        if (value!.isEmpty) {
                          return MyStrings.fieldErrorMsg.tr;
                        } else {
                          return null;
                        }
                      },
                      prefixIcon: Icon(
                        CupertinoIcons.search,
                        size: Dimensions.textFieldIconSize,
                      ),
                    ),
                  )
                : Container(
                    padding: EdgeInsetsDirectional.only(
                        start: Dimensions.space16.w,
                        end: Dimensions.space10.w,
                        top: Dimensions.space12.h,
                        bottom: Dimensions.space12.h),
                    color: MyColor.white,
                    child: Row(
                      children: [
                        Expanded(
                          child: LabelTextField(
                            labelText: "",
                            hideLabel: true,
                            hintText: MyStrings.search.tr,
                            controller: controller.couponCodeController,
                            hintTextColor: MyColor.getBodyTextColor(),
                            onChanged: (value) {
                              if (_debounceTimer?.isActive ?? false) {
                                _debounceTimer?.cancel();
                              }

                              _debounceTimer =
                                  Timer(const Duration(milliseconds: 500), () {
                                controller.getSaleCouponData();
                              });
                            },
                            textInputType: TextInputType.text,
                            textInputAction: TextInputAction.done,
                            radius: Dimensions.largeRadius,
                            validator: (value) {
                              if (value!.isEmpty) {
                                return MyStrings.fieldErrorMsg.tr;
                              } else {
                                return null;
                              }
                            },
                            prefixIcon: Icon(
                              CupertinoIcons.search,
                              size: Dimensions.textFieldIconSize,
                            ),
                          ),
                        ),
                        SizedBox(
                          width: Dimensions.space8.w,
                        ),
                        GestureDetector(
                          onTap: () {
                            if (controller.couponCodeController.text.isEmpty) {
                              CustomSnackBar.error(
                                  errorList: [MyStrings.enterCouponCode.tr]);
                            } else {
                              controller.applyCoupon(
                                  coupon: Coupon(
                                    code: controller.couponCodeController.text,
                                  ),
                                  isFromField: true);
                            }
                          },
                          child: Container(
                            padding: const EdgeInsetsDirectional.symmetric(
                                horizontal: Dimensions.space22,
                                vertical: Dimensions.space14),
                            decoration: BoxDecoration(
                                color: MyColor.getPrimaryColor(),
                                borderRadius: BorderRadius.circular(14)),
                            child: controller.isCouponLoading
                                ? SizedBox(
                                    width: 20,
                                    height: 20,
                                    child: CircularProgressIndicator(
                                        color: MyColor.white, strokeWidth: 2))
                                : Text(
                                    MyStrings.apply.tr,
                                    style: theme.textTheme.titleSmall?.copyWith(
                                        fontSize: 14, color: MyColor.white),
                                  ),
                          ),
                        ),
                      ],
                    ),
                  ),
            spaceDown(Dimensions.space10),
            controller.isLoading
                ? VariantShimmer()
                : controller.filteredCouponListData.isEmpty &&
                        controller.couponData.isEmpty
                    ? Center(
                        child: SizedBox(
                          height: MediaQuery.of(context).size.height * 0.5,
                          child: NoDataWidget(),
                        ),
                      )
                    : Expanded(
                        child: CustomAppCard(
                            radius: 0,
                            showBorder: false,
                            child: ListView.separated(
                                controller: scrollController,
                                physics: BouncingScrollPhysics(),
                                separatorBuilder: (context, index) =>
                                    CustomDivider(space: Dimensions.space16.h),
                                shrinkWrap: true,
                                itemCount: controller.isFromSale
                                    ? controller.couponData.length
                                    : controller.filteredCouponListData.length,
                                itemBuilder: (context, i) {
                                  if (!controller.isFromSale) {
                                    if (i ==
                                        controller
                                            .filteredCouponListData.length) {
                                      return controller.hasNext()
                                          ? SizedBox(
                                              height: Dimensions.space40,
                                              width: MediaQuery.of(context)
                                                  .size
                                                  .width,
                                              child: const CustomLoader(
                                                  isPagination: true),
                                            )
                                          : const SizedBox();
                                    }
                                  }

                                  return CouponListTile(
                                    index: i,
                                    isFromSale: widget.isFromSale,
                                  );
                                })),
                      )
          ],
        ),
        floatingActionButton: widget.isFromSale
            ? SizedBox.shrink()
            : CustomAppCard(
                onPressed: () {
                  if (Get.find<GlobalController>()
                      .checkPermission(featureName: AppPermissions.addCoupon)) {
                    Get.toNamed(RouteHelper.addNewCouponScreen)?.then((v) {
                      widget.isFromSale
                          ? controller.getSaleCouponData()
                          : controller.getCouponData();
                    });
                  }
                },
                height: Dimensions.space64.h,
                width: Dimensions.space64.w,
                padding: EdgeInsetsDirectional.all(Dimensions.space20),
                backgroundColor: MyColor.getPrimaryColor(),
                showBorder: false,
                child: MyAssetImageWidget(
                  color: theme.cardColor,
                  assetPath: MyIcons.add,
                  isSvg: true,
                )),
      ),
    );
  }
}
