import 'dart:async';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:ovosale/app/components/card/custom_app_card.dart';
import 'package:ovosale/app/components/card/my_custom_scaffold.dart';
import 'package:ovosale/app/components/custom_loader/custom_loader.dart';
import 'package:ovosale/app/components/divider/custom_divider.dart';
import 'package:ovosale/app/components/image/my_asset_widget.dart';
import 'package:ovosale/app/components/no_data.dart';
import 'package:ovosale/app/components/shimmer/brands_shimmer.dart';
import 'package:ovosale/app/components/text-field/label_text_field.dart';
import 'package:ovosale/app/screens/brands/widgets/brand_tile.dart';
import 'package:ovosale/core/route/route.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:ovosale/data/controller/brand/brand_controller.dart';
import 'package:ovosale/data/repo/brand/brand_repo.dart';

import '../../../core/app_permission/app_permission.dart';
import '../../../data/controller/cart/global_controller.dart';

class BrandsScreen extends StatefulWidget {
  const BrandsScreen({super.key});

  @override
  State<BrandsScreen> createState() => _BrandsScreenState();
}

class _BrandsScreenState extends State<BrandsScreen> {
  final ScrollController scrollController = ScrollController();

  void scrollListener() {
    if (scrollController.position.pixels ==
        scrollController.position.maxScrollExtent) {
      if (Get.find<BrandController>().hasNext()) {
        Get.find<BrandController>().getBrand(isPagination: true);
      }
    }
  }

  Timer? _debounceTimer;
  @override
  void initState() {
    Get.put(BrandRepo());
    final controller = Get.put(BrandController(brandRepo: Get.find()));
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.getBrand();
      scrollController.addListener(scrollListener);
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ThemeData theme = Theme.of(context);
    return GetBuilder<BrandController>(
      builder: (controller) => MyCustomScaffold(
        pageTitle: MyStrings.brands.tr,
        padding: EdgeInsets.zero,
        body: Column(
          children: [
            Container(
              padding: EdgeInsetsDirectional.symmetric(
                  horizontal: Dimensions.space16.w,
                  vertical: Dimensions.space12.h),
              color: MyColor.white,
              child: LabelTextField(
                labelText: "",
                hideLabel: true,
                hintText: MyStrings.search.tr,
                controller: controller.brandSearchController,
                hintTextColor: MyColor.getBodyTextColor(),
                onChanged: (value) {
                  if (_debounceTimer?.isActive ?? false) {
                    _debounceTimer?.cancel();
                  }

                  _debounceTimer = Timer(const Duration(milliseconds: 500), () {
                    controller.getBrand();
                  });
                },
                textInputType: TextInputType.text,
                textInputAction: TextInputAction.done,
                radius: Dimensions.largeRadius,
                validator: (value) {
                  if (value!.isEmpty) {
                    return MyStrings.fieldErrorMsg.tr;
                  } else {
                    return null;
                  }
                },
                prefixIcon: Icon(
                  CupertinoIcons.search,
                  size: Dimensions.textFieldIconSize,
                ),
              ),
            ),
            spaceDown(Dimensions.space10),
            controller.isLoading
                ? BrandsShimmer()
                : controller.filteredBrandDataList.isEmpty
                    ? Center(
                        child: SizedBox(
                          height: MediaQuery.of(context).size.height * 0.5,
                          child: NoDataWidget(),
                        ),
                      )
                    : Expanded(
                        child: CustomAppCard(
                            radius: 0,
                            showBorder: false,
                            child: ListView.separated(
                                controller: scrollController,
                                physics: const BouncingScrollPhysics(),
                                separatorBuilder: (context, index) =>
                                    CustomDivider(space: Dimensions.space16.h),
                                shrinkWrap: true,
                                itemCount:
                                    controller.filteredBrandDataList.length + 1,
                                itemBuilder: (context, i) {
                                  if (i ==
                                      controller.filteredBrandDataList.length) {
                                    return controller.hasNext()
                                        ? SizedBox(
                                            height: Dimensions.space40,
                                            width: MediaQuery.of(context)
                                                .size
                                                .width,
                                            child: const CustomLoader(
                                                isPagination: true),
                                          )
                                        : const SizedBox();
                                  }
                                  return BrandScreenListTile(
                                    index: i,
                                    brandDataList:
                                        controller.filteredBrandDataList,
                                  );
                                })),
                      )
          ],
        ),
        floatingActionButton: CustomAppCard(
          onPressed: () {
            if (Get.find<GlobalController>()
                .checkPermission(featureName: AppPermissions.addBrand)) {
              Get.toNamed(RouteHelper.newBrandScreen)?.then((v) {
                controller.getBrand();
              });
            }
          },
          height: Dimensions.space64.h,
          width: Dimensions.space64.w,
          padding: EdgeInsetsDirectional.all(Dimensions.space20),
          backgroundColor: MyColor.getPrimaryColor(),
          showBorder: false,
          child: MyAssetImageWidget(
            color: theme.cardColor,
            assetPath: MyIcons.add,
            isSvg: true,
          ),
        ),
      ),
    );
  }
}
