import 'dart:async';

import 'package:bluetooth_print_plus/bluetooth_print_plus.dart';
import 'package:flutter/services.dart';
import 'package:ovosale/core/utils/util_exporter.dart';
import 'package:rxdart/rxdart.dart' as prefix;

class BluetoothPrintService {
  static const int CONNECTED = 1;
  static const int DISCONNECTED = 0;

  BluetoothPrintService._();

  static final BluetoothPrintService _instance = BluetoothPrintService._();
  static BluetoothPrintService get instance => _instance;

  // Check if bluetooth is available
  Future<bool> get isAvailable async {
    try {
      return true; // Assume available if no exception
    } catch (e) {
      printX('Error checking bluetooth availability: $e');
      return false;
    }
  }

  // Check if bluetooth is turned on
  Future<bool> get isOn async {
    try {
      return BluetoothPrintPlus.isBlueOn;
    } catch (e) {
      printX('Error checking bluetooth state: $e');
      return false;
    }
  }

  // Check if device is connected
  Future<bool?> get isConnected async {
    try {
      return BluetoothPrintPlus.isConnected;
    } catch (e) {
      printX('Error checking connection status: $e');
      return false;
    }
  }

  final prefix.BehaviorSubject<bool> _isScanning = prefix.BehaviorSubject.seeded(false);
  Stream<bool> get isScanning => _isScanning.stream;

  final prefix.BehaviorSubject<List<BluetoothDevice>> _scanResults = prefix.BehaviorSubject.seeded([]);
  Stream<List<BluetoothDevice>> get scanResults => _scanResults.stream;

  final prefix.PublishSubject _stopScanPill = prefix.PublishSubject();

  /// Gets the current state of the Bluetooth module
  Stream<int> get state async* {
    try {
      yield* BluetoothPrintPlus.connectState.map((connectState) {
        switch (connectState) {
          case ConnectState.connected:
            return CONNECTED;
          case ConnectState.disconnected:
            return DISCONNECTED;
        }
      });
    } catch (e) {
      printX('Error getting bluetooth state: $e');
      yield DISCONNECTED;
    }
  }

  /// Enhanced scan method with better error handling
  Stream<BluetoothDevice> scan({Duration? timeout}) async* {
    if (_isScanning.value == true) {
      throw Exception('Another scan is already in progress.');
    }

    // Check if bluetooth is available and on
    if (!await isAvailable) {
      throw Exception('Bluetooth is not available on this device');
    }

    if (!await isOn) {
      throw Exception('Bluetooth is turned off. Please turn on Bluetooth.');
    }

    _isScanning.add(true);

    final killStreams = <Stream>[];
    killStreams.add(_stopScanPill);
    if (timeout != null) {
      killStreams.add(prefix.Rx.timer(null, timeout));
    }

    // Clear scan results
    _scanResults.add(<BluetoothDevice>[]);

    try {
      // Start scan using bluetooth_print_plus
      await BluetoothPrintPlus.startScan(timeout: timeout ?? Duration(seconds: 10));

      // Listen to scan results
      yield* BluetoothPrintPlus.scanResults.takeUntil(prefix.Rx.merge(killStreams)).doOnDone(() => stopScan()).expand((deviceList) => deviceList).map((device) {
        try {
          final List<BluetoothDevice> list = List.from(_scanResults.value);

          int existingIndex = list.indexWhere((e) => e.address == device.address);

          if (existingIndex != -1) {
            list[existingIndex] = device;
          } else {
            list.add(device);
          }

          _scanResults.add(list);
          return device;
        } catch (e) {
          printX('Error processing scan result: $e');
          throw Exception('Error processing scan result: $e');
        }
      });
    } on MissingPluginException catch (e) {
      _stopScanPill.add(null);
      _isScanning.add(false);
      throw Exception('Bluetooth plugin not properly configured. Error: ${e.message}');
    } on PlatformException catch (e) {
      _stopScanPill.add(null);
      _isScanning.add(false);

      String errorMessage = 'Platform error: ${e.message}';
      switch (e.code) {
        case 'BLUETOOTH_NOT_AVAILABLE':
          errorMessage = 'Bluetooth is not available on this device';
          break;
        case 'BLUETOOTH_NOT_ENABLED':
          errorMessage = 'Please enable Bluetooth';
          break;
        case 'LOCATION_PERMISSION_DENIED':
          errorMessage = 'Location permission required for Bluetooth scanning';
          break;
        case 'ALREADY_SCANNING':
          errorMessage = 'Bluetooth scan already in progress';
          break;
      }
      throw Exception(errorMessage);
    } catch (e) {
      printX('Error starting scan: $e');
      _stopScanPill.add(null);
      _isScanning.add(false);
      throw Exception('Failed to start bluetooth scan: $e');
    }
  }

  /// Start scan and return devices when completed
  Future<List<BluetoothDevice>> startScan({Duration? timeout}) async {
    try {
      await scan(timeout: timeout).drain();
      return _scanResults.value;
    } catch (e) {
      printX('Error in startScan: $e');
      rethrow;
    }
  }

  /// Stop bluetooth scan
  Future<void> stopScan() async {
    try {
      await BluetoothPrintPlus.stopScan();
      _stopScanPill.add(null);
      _isScanning.add(false);
    } catch (e) {
      printX('Error stopping scan: $e');
      _stopScanPill.add(null);
      _isScanning.add(false);
    }
  }

  /// Connect to bluetooth device
  Future<dynamic> connect(BluetoothDevice device) async {
    try {
      return await BluetoothPrintPlus.connect(device);
    } on MissingPluginException catch (e) {
      throw Exception('Connect method not implemented in native plugin: ${e.message}');
    } catch (e) {
      printX('Error connecting to device: $e');
      rethrow;
    }
  }

  /// Disconnect from bluetooth device
  Future<dynamic> disconnect() async {
    try {
      return await BluetoothPrintPlus.disconnect();
    } catch (e) {
      printX('Error disconnecting: $e');
      rethrow;
    }
  }

  /// Print receipt using raw commands
  Future<bool> printReceipt(Map<String, dynamic> config, List<LineText> data) async {
    try {
      List<int> bytes = [];

      for (var lineText in data) {
        switch (lineText.type) {
          case LineText.TYPE_TEXT:
            // Add ESC/POS text commands
            String text = lineText.content ?? '';
            bytes.addAll(text.codeUnits);
            bytes.addAll([0x0A]); // Line feed
            break;
          case LineText.TYPE_BARCODE:
            // Add barcode commands if needed
            break;
          case LineText.TYPE_QRCODE:
            // Add QR code commands if needed
            break;
        }
      }

      // Add cut command
      bytes.addAll([0x1D, 0x56, 0x41, 0x10]);

      await BluetoothPrintPlus.write(Uint8List.fromList(bytes));
      return true;
    } on MissingPluginException catch (e) {
      throw Exception('Print receipt method not implemented: ${e.message}');
    } catch (e) {
      printX('Error printing receipt: $e');
      return false;
    }
  }

  /// Safe scan method with retry logic
  Future<List<BluetoothDevice>> safeScan({
    Duration timeout = const Duration(seconds: 10),
    int maxRetries = 3,
  }) async {
    for (int attempt = 0; attempt < maxRetries; attempt++) {
      try {
        printX('Bluetooth scan attempt ${attempt + 1}/$maxRetries');

        // Check prerequisites
        if (!await isAvailable) {
          throw Exception('Bluetooth not available on this device');
        }

        if (!await isOn) {
          throw Exception('Bluetooth is turned off');
        }

        return await startScan(timeout: timeout);
      } on MissingPluginException {
        throw Exception('Bluetooth plugin not properly configured. Please restart the app and ensure the plugin is correctly installed.');
      } catch (e) {
        printX('Scan attempt ${attempt + 1} failed: $e');

        if (attempt == maxRetries - 1) {
          throw Exception('Failed to scan after $maxRetries attempts: $e');
        }

        // Wait before retry
        await Future.delayed(Duration(seconds: 1));
      }
    }

    return [];
  }
}

// Helper class for LineText compatibility
class LineText {
  static const int TYPE_TEXT = 0;
  static const int TYPE_BARCODE = 1;
  static const int TYPE_QRCODE = 2;
  static const int TYPE_IMAGE = 3;

  final int type;
  final String? content;
  final int? x;
  final int? y;
  final int? size;
  final int? align;

  LineText({
    required this.type,
    this.content,
    this.x,
    this.y,
    this.size,
    this.align,
  });

  Map<String, dynamic> toJson() {
    return {
      'type': type,
      'content': content,
      'x': x,
      'y': y,
      'size': size,
      'align': align,
    };
  }

  factory LineText.fromJson(Map<String, dynamic> json) {
    return LineText(
      type: json['type'],
      content: json['content'],
      x: json['x'],
      y: json['y'],
      size: json['size'],
      align: json['align'],
    );
  }
}
