package com.ovosolution.ovosale

import android.Manifest
import android.app.Activity
import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothDevice
import android.bluetooth.BluetoothManager
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import io.flutter.embedding.engine.plugins.FlutterPlugin
import io.flutter.embedding.engine.plugins.activity.ActivityAware
import io.flutter.embedding.engine.plugins.activity.ActivityPluginBinding
import io.flutter.plugin.common.EventChannel
import io.flutter.plugin.common.MethodCall
import io.flutter.plugin.common.MethodChannel
import io.flutter.plugin.common.MethodChannel.MethodCallHandler
import io.flutter.plugin.common.MethodChannel.Result

class BluetoothPrintPlugin: FlutterPlugin, MethodCallHandler, ActivityAware {
    private lateinit var channel: MethodChannel
    private lateinit var eventChannel: EventChannel
    private var context: Context? = null
    private var activity: Activity? = null
    private var bluetoothAdapter: BluetoothAdapter? = null
    private var eventSink: EventChannel.EventSink? = null
    private var isScanning = false

    companion object {
        private const val CHANNEL_NAME = "bluetooth_print/methods"
        private const val EVENT_CHANNEL_NAME = "bluetooth_print/state"
        private const val REQUEST_ENABLE_BT = 1
        private const val REQUEST_PERMISSIONS = 2
    }

    override fun onAttachedToEngine(flutterPluginBinding: FlutterPlugin.FlutterPluginBinding) {
        channel = MethodChannel(flutterPluginBinding.binaryMessenger, CHANNEL_NAME)
        channel.setMethodCallHandler(this)
        
        eventChannel = EventChannel(flutterPluginBinding.binaryMessenger, EVENT_CHANNEL_NAME)
        eventChannel.setStreamHandler(object : EventChannel.StreamHandler {
            override fun onListen(arguments: Any?, events: EventChannel.EventSink?) {
                eventSink = events
            }

            override fun onCancel(arguments: Any?) {
                eventSink = null
            }
        })
        
        context = flutterPluginBinding.applicationContext
        
        val bluetoothManager = context?.getSystemService(Context.BLUETOOTH_SERVICE) as BluetoothManager
        bluetoothAdapter = bluetoothManager.adapter
    }

    override fun onMethodCall(call: MethodCall, result: Result) {
        when (call.method) {
            "isAvailable" -> {
                result.success(bluetoothAdapter != null)
            }
            "isOn" -> {
                result.success(bluetoothAdapter?.isEnabled == true)
            }
            "isConnected" -> {
                // Implement connection status check
                result.success(false) // Placeholder
            }
            "state" -> {
                val state = when {
                    bluetoothAdapter == null -> 10 // STATE_OFF
                    bluetoothAdapter!!.isEnabled -> 12 // STATE_ON
                    else -> 10 // STATE_OFF
                }
                result.success(state)
            }
            "startScan" -> {
                startScan(result)
            }
            "stopScan" -> {
                stopScan(result)
            }
            "connect" -> {
                // Implement device connection
                result.success(true) // Placeholder
            }
            "disconnect" -> {
                // Implement device disconnection
                result.success(true) // Placeholder
            }
            "destroy" -> {
                // Implement cleanup
                result.success(true) // Placeholder
            }
            "printReceipt" -> {
                // Implement receipt printing
                result.success(true) // Placeholder
            }
            "printLabel" -> {
                // Implement label printing
                result.success(true) // Placeholder
            }
            "printTest" -> {
                // Implement test printing
                result.success(true) // Placeholder
            }
            else -> {
                result.notImplemented()
            }
        }
    }

    private fun startScan(result: Result) {
        if (!hasPermissions()) {
            requestPermissions()
            result.error("NO_PERMISSIONS", "Bluetooth permissions not granted", null)
            return
        }

        if (bluetoothAdapter == null || !bluetoothAdapter!!.isEnabled) {
            result.error("BLUETOOTH_OFF", "Bluetooth is not enabled", null)
            return
        }

        if (isScanning) {
            result.error("ALREADY_SCANNING", "Scan already in progress", null)
            return
        }

        try {
            isScanning = true
            val filter = IntentFilter(BluetoothDevice.ACTION_FOUND)
            context?.registerReceiver(scanReceiver, filter)
            
            if (ActivityCompat.checkSelfPermission(context!!, Manifest.permission.BLUETOOTH_SCAN) != PackageManager.PERMISSION_GRANTED) {
                result.error("NO_PERMISSIONS", "Missing BLUETOOTH_SCAN permission", null)
                return
            }
            bluetoothAdapter!!.startDiscovery()
            result.success(true)
        } catch (e: Exception) {
            isScanning = false
            result.error("SCAN_ERROR", e.message, null)
        }
    }

    private fun stopScan(result: Result) {
        try {
            if (isScanning) {
                context?.unregisterReceiver(scanReceiver)
                if (ActivityCompat.checkSelfPermission(context!!, Manifest.permission.BLUETOOTH_SCAN) == PackageManager.PERMISSION_GRANTED) {
                    bluetoothAdapter?.cancelDiscovery()
                }
                isScanning = false
            }
            result.success(true)
        } catch (e: Exception) {
            result.error("STOP_SCAN_ERROR", e.message, null)
        }
    }

    private val scanReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            when (intent.action) {
                BluetoothDevice.ACTION_FOUND -> {
                    val device: BluetoothDevice? = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE)
                    device?.let {
                        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.BLUETOOTH_CONNECT) == PackageManager.PERMISSION_GRANTED) {
                            val deviceInfo = mapOf(
                                "name" to (it.name ?: "Unknown"),
                                "address" to it.address,
                                "type" to it.type,
                                "connected" to false
                            )
                            channel.invokeMethod("ScanResult", deviceInfo)
                        }
                    }
                }
            }
        }
    }

    private fun hasPermissions(): Boolean {
        val permissions = arrayOf(
            Manifest.permission.BLUETOOTH,
            Manifest.permission.BLUETOOTH_ADMIN,
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.BLUETOOTH_SCAN,
            Manifest.permission.BLUETOOTH_CONNECT
        )
        
        return permissions.all { permission ->
            ContextCompat.checkSelfPermission(context!!, permission) == PackageManager.PERMISSION_GRANTED
        }
    }

    private fun requestPermissions() {
        val permissions = arrayOf(
            Manifest.permission.BLUETOOTH,
            Manifest.permission.BLUETOOTH_ADMIN,
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.BLUETOOTH_SCAN,
            Manifest.permission.BLUETOOTH_CONNECT
        )
        
        activity?.let {
            ActivityCompat.requestPermissions(it, permissions, REQUEST_PERMISSIONS)
        }
    }

    override fun onDetachedFromEngine(binding: FlutterPlugin.FlutterPluginBinding) {
        channel.setMethodCallHandler(null)
        eventChannel.setStreamHandler(null)
    }

    override fun onAttachedToActivity(binding: ActivityPluginBinding) {
        activity = binding.activity
    }

    override fun onDetachedFromActivityForConfigChanges() {
        activity = null
    }

    override fun onReattachedToActivityForConfigChanges(binding: ActivityPluginBinding) {
        activity = binding.activity
    }

    override fun onDetachedFromActivity() {
        activity = null
    }
}